/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "where.h"
#include "dfile_cache_create.h"


/*
** This function makes an initial pass through the data to calculate
** how much data will be loaded into shared memory. It also finds maximum
** size of each field.
*/
int get_data_size( size_t *record_size, size_t *record_cnt, size_t **field_size, dfile_t *dfile, void *input_filter )
{
	size_t	*max_field_size, alloc_size, rec_cnt;
	unsigned short	ndx, bind_cnt;
	dfile_bind_t	*bind;
	where_result_t	where_result;
	char	err_msg[ 256 ];

	assert( record_size != (size_t *)0 );
	assert( record_cnt != (size_t *)0 );
	assert( field_size != (size_t **)0 );
	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	bind_cnt = dfile->bind_cnt;
	bind = dfile->bind;

	alloc_size = (size_t)bind_cnt * sizeof( size_t );
	max_field_size = (size_t *)malloc( alloc_size );
	if ( max_field_size == (size_t *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	(void) memset( (void *)max_field_size, 0, alloc_size );
	rec_cnt = (size_t)0;

	while ( dfile_read( dfile ) == 0 ) {
		if ( input_filter != (void *)0 ) {
			if ( where_condition( &where_result, err_msg, sizeof( err_msg ), input_filter ) != WHERE_NOERR ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "input where_condition() failed [", stderr );
				(void) fputs( err_msg, stderr );
				(void) fputs( "].\n", stderr );
				RETURN_INT( -1 );
			}
			if ( where_result == Where_result_false ) {
				/*
				** Skip record.
				*/
				continue;
			}
		}

		++rec_cnt;
		for ( ndx = (unsigned short)0; ndx < bind_cnt; ++ndx ) {
			if ( *bind[ ndx ].field_length > max_field_size[ ndx ] ) {
				max_field_size[ ndx ] = *bind[ ndx ].field_length;
			}
		}
	}

	if ( dfile->error != Dfile_all_data_processed ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to read all data.\n", stderr );
		RETURN_INT( -1 );
	}

	*record_size = (size_t)0;

	for ( ndx = (unsigned short)0; ndx < bind_cnt; ++ndx ) {
		/*
		** Add one byte for null termination.
		*/
		++max_field_size[ ndx ];
		*record_size += max_field_size[ ndx ];
	}

	*record_cnt = rec_cnt;
	*field_size = max_field_size;

	if ( Debug ) {
		(void) fprintf( stderr, "record count is %u\n", *record_cnt );
		(void) fprintf( stderr, "record size is %u\n", *record_size );
		for ( ndx = (unsigned short)0; ndx < bind_cnt; ++ndx ) {
			(void) fprintf( stderr, "field [%s] size is %u\n", bind[ ndx ].field_name, ( *field_size )[ ndx ] );
		}
	}

	RETURN_INT( 0 );
}
