/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <regex.h>
#include "tbox.h"
#include "sexpr.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "where.h"
#include "_where.h"

static const char       rcsid[] = "$Id: _where_like_syntax.c,v 1.2 2009/10/16 17:27:11 keith Exp $";

/*
** $Log: _where_like_syntax.c,v $
** Revision 1.2  2009/10/16 17:27:11  keith
** Added GPL to source code.
**
** Revision 1.1  2009/02/18 06:06:40  keith
** Initial revision
**
*/

/*
** This function builds LIKE expression portion of syntax tree.
*/

int _where_like_syntax( like_cond_t *like_cond, sexpr_t *sexpr, dfile_t *dfile )
{
	static const char func[] = "_where_like_syntax";
	int	status;
	char	*variable_name;
	dfile_bind_t	bind, **bind_entry;
	regex_t	*ex;
	char	msg[256];

	assert( like_cond != (like_cond_t *)0 );
	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	assert( SEXPR_CAR_TYPE( sexpr ) == string_sexpr );
	variable_name = (char *)SEXPR_CAR_STRING( sexpr );

	if ( *variable_name != '$' ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Expected variable but found [", stderr );
		(void) fputs( variable_name, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	++variable_name;
	strtoupper( variable_name );
	bind.field_name = (const char *)variable_name;

	bind_entry = (dfile_bind_t **)dhfind( (void *)&bind, dfile->bind_hash_table, strhkey( bind.field_name ), dfile_bind_field_name_cmp );

	if ( bind_entry == (dfile_bind_t **)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Variable [", stderr );
		(void) fputs( variable_name, stderr );
		(void) fputs( "] was not defined.\n", stderr );
		RETURN_INT( WHERE_UNKNVAR );
	}

	like_cond->dfile_bind = *bind_entry;

	if ( SEXPR_CDR_TYPE( sexpr ) != list_sexpr ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Expected pattern to follow variable [", stderr );
		(void) fputs( variable_name, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	sexpr = SEXPR_CDR_LIST( sexpr );

	if ( sexpr == (sexpr_t *)0 || SEXPR_CAR_TYPE( sexpr ) != string_sexpr ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Expected pattern string to follow variable [", stderr );
		(void) fputs( variable_name, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	ex = (regex_t *)malloc( sizeof( regex_t ) );
	if ( ex == (regex_t *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( WHERE_UNIXERR );
	}

	if ( Debug ) {
		(void) fprintf( stderr, "compiling regex [%s]\n", SEXPR_CAR_STRING( sexpr ) );
	}

	status = regcomp( ex, SEXPR_CAR_STRING( sexpr ), REG_EXTENDED|REG_NOSUB );
	if ( status != 0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to compile pattern [", stderr );
		(void) fputs( SEXPR_CAR_STRING( sexpr ), stderr );
		(void) fputs( "], [", stderr );
		(void) regerror( status, ex, msg, sizeof( msg ) );
		(void) fputs( msg, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( WHERE_REGEX );
	}

	like_cond->regex = ex;

	if ( SEXPR_CDR_LIST( sexpr ) != (sexpr_t *)0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Too many arguments for LIKE operator.\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	RETURN_INT( WHERE_NOERR );
}
