/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"

static const char       rcsid[] = "$Id: insertion_sort.c,v 1.2 2009/10/16 18:00:43 keith Exp $";

/*
** $Log: insertion_sort.c,v $
** Revision 1.2  2009/10/16 18:00:43  keith
** Added GPL to source code.
**
** Revision 1.1  2009/02/14 18:01:10  keith
** Initial revision
**
*/

/*
** This function sorts an array using insertion sort.
*/

int insertion_sort( void *base, size_t element_cnt, size_t element_size, int ( *cmp )(const void *, const void * ) )
{
	static const char func[] = "insertion_sort";
	char	*end;
	void 	*save;

	assert( base != (void *)0 );
	assert( cmp != (int (*)(const void *, const void * ) )0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( %p, %u, %u, %p )\n", func, base, element_cnt, element_size, cmp );
	}

	DEBUG_FUNC_START;

	if ( element_cnt < (size_t)2 ) {
		if ( Debug ) {
			(void) fputs( "Not enough elements to sort.\n", stderr );
		}
		RETURN_INT( 0 );
	}

	if ( element_size == (size_t)0 ) {
		if ( Debug ) {
			(void) fputs( "Element size was zero.\n", stderr );
		}
		RETURN_INT( -1 );
	}

	end = (char *)base + ( element_cnt * element_size );

	save = malloc( element_size );
	if ( save == (void *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	internal_insertion_sort( base, element_cnt, element_size, cmp, save );

	free( save );

	RETURN_INT( 0 );
}

void internal_insertion_sort( void *base, size_t element_cnt, size_t element_size, int ( *cmp )(const void *, const void * ), void *save )
{
	static const char func[] = "internal_insertion_sort";
	char	*right, *left, *next_left, *end;

	assert( base != (void *)0 );
	assert( cmp != (int (*)(const void *, const void * ) )0 );
	assert( save != (void *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( %p, %u, %u, %p, %p )\n", func, base, element_cnt, element_size, cmp, save );
	}

	DEBUG_FUNC_START;

	end = (char *)base + ( element_cnt * element_size );
	right = ( char *)base + element_size;

	while ( end > right ) {
		(void) memcpy( save, (void *)right, element_size );
		left = right;
		next_left = left - element_size;
		while ( next_left >= (char *)base && ( *cmp )( (void *)next_left, save ) > 0 ) {
			(void) memcpy( (void *)left, (void *)next_left, element_size );
			left = next_left;
			next_left -= element_size;
		}
		if ( right > left ) {
			(void) memcpy( (void *)left, save, element_size );
		}
		right += element_size;
	}

	RETURN_VOID;
}

#ifdef MT_insertion_sort

#include <stdlib.h>
/*
** This function is used to regression test insertion_sort().
** The following command is used to compile:
**   x=insertion_sort; make "MT_CC=-DMT_$x" $x
*/

#define	SORT_CNT	10

int long_cmp( const void *x, const void *y )
{
	return *(long *)x - *(long *)y;
}

int main( void )

{
	static const char	complete_msg[] =  ">>> Module test on function %s() is complete.\n";
	static const char	test_func[] = "insertion_sort";
	static const char	successful[] = ">>>\n>>> %s() was successful.\n";
	static const char	unsuccessful[] = ">>>\n>>> %s() was unsuccessful.\n";
	static const char	blank_line[] = ">>>\n";
	long	x[SORT_CNT];
	unsigned short	ndx;
	int	ret;

	Debug = 1;

	for ( ndx = (unsigned short)0; ndx < SORT_CNT; ++ndx ) {
		x[ ndx ] = random() % 10000;
	}

	(void) fprintf( stderr, ">>> Start module test on function %s().\n", test_func );
	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #1\n", stderr );
	(void) fputs( ">>> Sort the following integers.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) fputs( "UNSORTED:\n", stderr );
	for ( ndx = (unsigned short)0; ndx < SORT_CNT; ++ndx ) {
		(void) fprintf( stderr, "%7ld", x[ ndx ] );
	}
	(void) fputc( '\n', stderr );

	ret = insertion_sort( (void *)x, SORT_CNT, sizeof( long ), long_cmp );
	if ( ret == -1 ) {
		(void) fprintf( stderr, unsuccessful, test_func );
		return 1;
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( "SORTED:\n", stderr );
	for ( ndx = (unsigned short)0; ndx < SORT_CNT; ++ndx ) {
		(void) fprintf( stderr, "%7ld", x[ ndx ] );
	}
	(void) fputc( '\n', stderr );

	for ( ndx = (unsigned short)0; ndx < ( SORT_CNT - 1 ); ++ndx ) {
		if ( x[ ndx ] > x[ ndx + 1 ] ) {
			(void) fprintf( stderr, unsuccessful, test_func );
			return 1;
		}
	}

	(void) fprintf( stderr, successful, test_func );

	(void) fputs( blank_line, stderr );
	(void) fprintf( stderr, complete_msg, test_func );
	exit( 0 );
}
#endif
