/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <assert.h>
#include "tbox.h"

static const char       rcsid[] = "$Id: crc16.c,v 1.2 2009/10/16 18:00:43 keith Exp $";

/*
** $Log: crc16.c,v $
** Revision 1.2  2009/10/16 18:00:43  keith
** Added GPL to source code.
**
** Revision 1.1  2009/02/14 09:11:54  keith
** Initial revision
**
*/

/*
** This 16-bit CRC (cyclic redundency check) function was borrowed from
** Numerical Recipes in C, Second Edition.
*/

static unsigned short	icrc1(unsigned short crc, unsigned char onech);

typedef unsigned char uchar;
#define LOBYTE(x) ((uchar)((x) & 0xFF))
#define HIBYTE(x) ((uchar)((x) >> 8))

unsigned short crc16(unsigned short crc, const unsigned char *bufptr, size_t len, short jinit, int jrev)
{
	static const char func[] = "crc16";
	static const char end_msg[] = "End %s() returning %hx at %s\n";
	static unsigned short icrctb[256],init=0;
	static uchar rchr[256];
	unsigned short j,cword=crc;
	unsigned short result;
	static uchar it[16]={0,8,4,12,2,10,6,14,1,9,5,13,3,11,7,15};

	assert( bufptr != (unsigned char *)0 );

	DEBUG_FUNC_START;

	if (!init) {
		init=1;
		for (j=0;j<=255;j++) {
			icrctb[j]=icrc1(j << 8,(uchar)0);
			rchr[j]=(uchar)(it[j & 0xF] << 4 | it[j >> 4]);
		}
	}
	if (jinit >= 0) cword=((uchar) jinit) | (((uchar) jinit) << 8);
	else if (jrev < 0) cword=rchr[HIBYTE(cword)] | rchr[LOBYTE(cword)] << 8;
	for (j=0;j<len;j++)
		cword=icrctb[(jrev < 0 ? rchr[bufptr[j]] :
			bufptr[j]) ^ HIBYTE(cword)] ^ LOBYTE(cword) << 8;

	if ( jrev >= 0 ) {
		result = cword;
	} else {
		result = rchr[HIBYTE(cword)] | rchr[LOBYTE(cword)] << 8;
	}

	RETURN_INT( result );
}
#undef LOBYTE
#undef HIBYTE

static unsigned short icrc1(unsigned short crc, unsigned char onech)
{
	unsigned short	i;
	unsigned short	ans=(crc ^ onech << 8);

	for (i=0;i<8;i++) {
		if (ans & 0x8000)
			ans = (ans <<= 1) ^ 4129;
		else
			ans <<= 1;
	}
	return ans;
}

#ifdef MT_crc16

/*
** This function is used to regression test crc16().
** The following command is used to compile:
**   x=crc16; make "MT_CC=-DMT_$x" "MT_PRE=DEFINE=MT_$x" $x
*/
int main( void )

{
	static const char	complete_msg[] =  ">>> Module test on function %s() is complete.\n";
	static const char	test_func[] = "crc16";
	static const char	successful[] = ">>>\n>>> %s() was successful.\n";
	static const char	unsuccessful[] = ">>>\n>>> %s() was unsuccessful.\n";
	static const char	blank_line[] = ">>>\n";
	unsigned short	crc;
	char	*str;

	Debug = 1;

	(void) fprintf( stderr, ">>> Start module test on function %s().\n", test_func );
	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #1\n", stderr );
	(void) fputs( ">>> Check XMODEM CRC for value 'T'.\n", stderr );
	(void) fputs( blank_line, stderr );

	str = "T";
	crc = crc16( (unsigned short)0, (unsigned char *)str, strlen( str ), (short)0, 1 );
	if ( crc == 0X1A71 ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #2\n", stderr );
	(void) fputs( ">>> Check XMODEM CRC for value 'CatMouse987654321'.\n", stderr );
	(void) fputs( blank_line, stderr );

	str = "CatMouse987654321";
	crc = crc16( (unsigned short)0, (unsigned char *)str, strlen( str ), (short)0, 1 );
	if ( crc == 0XE556 ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #3\n", stderr );
	(void) fputs( ">>> Check X.25 CRC for value 'T'.\n", stderr );
	(void) fputs( blank_line, stderr );

	str = "T";
	crc = crc16( (unsigned short)0, (unsigned char *)str, strlen( str ), (short)255, -1 );
	if ( crc == 0X1B26 ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #4\n", stderr );
	(void) fputs( ">>> Check X.25 CRC for value 'CatMouse987654321'.\n", stderr );
	(void) fputs( blank_line, stderr );

	str = "CatMouse987654321";
	crc = crc16( (unsigned short)0, (unsigned char *)str, strlen( str ), (short)255, -1 );
	if ( crc == 0XF56E ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #5\n", stderr );
	(void) fputs( ">>> Check CRC-CCITT for value 'T'.\n", stderr );
	(void) fputs( blank_line, stderr );

	str = "T";
	crc = crc16( (unsigned short)0, (unsigned char *)str, strlen( str ), (short)0, -1 );
	if ( crc == 0X14A1 ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #6\n", stderr );
	(void) fputs( ">>> Check CRC-CCITT for value 'CatMouse987654321'.\n", stderr );
	(void) fputs( blank_line, stderr );

	str = "CatMouse987654321";
	crc = crc16( (unsigned short)0, (unsigned char *)str, strlen( str ), (short)0, -1 );
	if ( crc == 0XC28D ) {
		(void) fprintf( stderr, successful, test_func );
	} else {
		(void) fprintf( stderr, unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fprintf( stderr, complete_msg, test_func );
	exit( 0 );
}
#endif
