/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include <zlib.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"

static const char       rcsid[] = "$Id: _dfile_write_gz_header.c,v 1.2 2009/10/16 07:35:57 keith Exp $";

/*
** $Log: _dfile_write_gz_header.c,v $
** Revision 1.2  2009/10/16 07:35:57  keith
** Add GPL to source code.
**
** Revision 1.1  2009/02/28 06:46:22  keith
** Initial revision
**
*/

#define	MAGIC_OFFSET	0
#define	METHOD_OFFSET	2
#define	FLAGS_OFFSET	3
#define	TIME_OFFSET	4
#define	XFLAGS_OFFSET	8
#define	OS_OFFSET	9
#define	XLEN_OFFSET	10

/* gzip flag byte */
#define ASCII_FLAG	0x01 /* bit 0 set: file probably ascii text */
#define HEAD_CRC	0x02 /* bit 1 set: header CRC present */
#define EXTRA_FIELD	0x04 /* bit 2 set: extra field present */
#define ORIG_NAME	0x08 /* bit 3 set: original file name present */
#define COMMENT		0x10 /* bit 4 set: file comment present */
#define RESERVED	0xE0 /* bits 5..7: reserved */

#define	DEFLATE_METHOD	8

/*
** This function copies a GZIP head to I/O buffer. It is expected to be
** at the beginning of file, and smaller than size of I/O buffer.
*/

void _dfile_write_gz_header( dfile_t *dfile )
{
	static const char	func[] = "_dfile_write_gz_header";
	static const char	comment[] = "CREATED USING DFILE";
	const size_t	comment_length = sizeof( comment );
	static const unsigned char	gz_magic[2] = {0x1f, 0x8b};
	size_t	header_length;
	unsigned char	*buf;
	z_stream	*zstream;
	time_t	gmt;
	size_t	fname_length;

	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	zstream = (z_stream *)dfile->zstream;
	assert( zstream != (z_stream *)0 );
	buf = (unsigned char *)zstream->next_out;
	assert( buf != (unsigned char *)0 );

	assert ( zstream->avail_out > 512U );

	(void) memcpy( (void *)&buf[ MAGIC_OFFSET ], (void *)gz_magic, sizeof( gz_magic ) );

	buf[ METHOD_OFFSET ] = (unsigned char)DEFLATE_METHOD;

	buf[ FLAGS_OFFSET ] = ASCII_FLAG | ORIG_NAME | COMMENT;

	(void) time( &gmt );
	(void) memcpy( (void *)&buf[ TIME_OFFSET ], &gmt, sizeof( gmt ) );

	buf[ XFLAGS_OFFSET ] = (unsigned char)0;

	/*
	** Hard code OS type as UNIX.
	*/
	buf[ OS_OFFSET ] = (unsigned char)3;

	header_length = OS_OFFSET + 1;

	fname_length = strlen( (const char *)dfile->file_name );
	assert( fname_length > (size_t)3 );
	/*
	** Truncate .gz suffix.
	*/
	fname_length -= (size_t)3;
	(void) memcpy( (void *)&buf[ header_length ], (void *)dfile->file_name, fname_length );

	header_length += fname_length;
	buf[ header_length ] = (unsigned char)0;
	++header_length;

	(void) memcpy( (void *)&buf[ header_length ], (void *)comment, comment_length );

	header_length += comment_length;

	zstream->next_out = (Bytef *)&buf[ header_length ];
	zstream->avail_out -= (unsigned int)header_length;

	RETURN_VOID;
}
