/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <assert.h>
#include <zlib.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"

static const char       rcsid[] = "$Id: _dfile_read_open_file.c,v 1.2 2009/10/16 07:35:56 keith Exp $";

/*
** $Log: _dfile_read_open_file.c,v $
** Revision 1.2  2009/10/16 07:35:56  keith
** Add GPL to source code.
**
** Revision 1.1  2009/02/27 08:00:19  keith
** Initial revision
**
*/

static int zip_init( dfile_t * );

/*
** This function opens a file for reading.
*/

int _dfile_read_open_file( dfile_t *dfile )
{
	static const char	func[] = "_dfile_read_open_file";
	int	fd;
	const char	*fname;
	dfile_buffer_t	*buffer;

	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	buffer = dfile->io_buffer;
	assert( buffer != (dfile_buffer_t *)0 );

	fname = dfile->file_name;

	assert( fname != (const char *)0 );

	if ( strcmp( fname, "-" ) == 0 ) {
		/*
		** Standard input.
		*/
		fd = 0;
	} else {
		fd = open( fname, O_RDONLY );
		if ( fd < 0 ) {
			UNIX_ERROR( "open() failed" );
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Failed to open file [", stderr );
			(void) fputs( fname, stderr );
			(void) fputs( "] for reading.\n", stderr );
			dfile->error = Dfile_open_failed;
			RETURN_INT( -1 );
		}
	}

	dfile->open_descriptor = fd;

	if ( buffer->buffer_area == (char *)0 ) {
		if ( _dfile_allocate_buffer( dfile, fname ) == -1 ) {
			RETURN_INT( -1 );
		}

		do {
			assert( buffer != (dfile_buffer_t *)0 );
			buffer->buf_ptr = buffer->buffer_end;
			buffer->buf_ptr_end = buffer->buf_ptr;
			buffer = buffer->next;
		} while ( buffer != dfile->buffer_base );
	}

	if ( dfile->format == Dfile_zipped ) {
		if ( zip_init( dfile ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	if ( _dfile_set_refill_buffer_func( dfile ) == -1 ) {
		RETURN_INT( -1 );
	}

	if ( Debug ) {
		(void) fprintf( stderr, "opened file [%s]\n", fname );
	}

	RETURN_INT( 0 );
}

static int zip_init( dfile_t *dfile )
{
	if ( dfile->zipped_buffer == (unsigned char *)0 ) {
		assert( dfile->zstream == (z_stream *)0 );
		if ( _dfile_allocate_zipped_stream( dfile, 'r' ) == -1 ) {
			return -1;
		}
	}

	assert( dfile->zipped_buffer != (unsigned char *)0 );
	assert( dfile->zstream != (z_stream *)0 );

	if ( _dfile_read_gz_header( dfile ) == -1 ) {
		return -1;
	}

	dfile->crc = crc32( 0UL, (const Bytef *)0, 0U );
	dfile->file_char_cnt = 0UL;

	return 0;
}
