/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"

static const char       rcsid[] = "$Id: _dfile_create_bind_hash_table.c,v 1.2 2009/10/16 07:35:54 keith Exp $";

/*
** $Log: _dfile_create_bind_hash_table.c,v $
** Revision 1.2  2009/10/16 07:35:54  keith
** Add GPL to source code.
**
** Revision 1.1  2009/02/22 13:16:34  keith
** Initial revision
**
*/

static unsigned long calc_hash( const void * );

/*
** This function creates a hash table based on dfile_bind_t structure
** field_name value.
*/

int _dfile_create_bind_hash_table( void **ret_hash_table, dfile_bind_t *bind_tbl, unsigned short bind_tbl_cnt )
{
	static const char	func[] = "_dfile_create_bind_hash_table";
	const unsigned long	max_collisions_allowed = 3;
	unsigned long	initial_slot_cnt;
	unsigned short	ndx;
	void	*hash_table;
	dfile_bind_t	**bind_entry;
	dherrno_t	dherrno;

	assert( ret_hash_table != (void **)0 );
	assert( bind_tbl != (dfile_bind_t *)0 );
	assert( bind_tbl_cnt > (unsigned short)0 );

	DEBUG_FUNC_START;

	*ret_hash_table = (void *)0;

	initial_slot_cnt = (unsigned long)bind_tbl_cnt * 3UL;

	hash_table = dhcreate( initial_slot_cnt, max_collisions_allowed );
	if ( hash_table == (void *)0 ) {
		UNIX_ERROR( "dhcreate() failed to allocate memory" );
		RETURN_INT( -1 );
	}

	for ( ndx = (unsigned short)0; ndx < bind_tbl_cnt; ++ndx ) {
		if ( Debug ) {
			(void) fprintf( stderr, "adding field [%s], %lu\n", bind_tbl->field_name, strhkey( bind_tbl->field_name ) );
		}
		bind_entry = (dfile_bind_t **)dhesearch( (const void *)bind_tbl, &hash_table, calc_hash, dfile_bind_field_name_cmp );

		if ( bind_entry == (dfile_bind_t **)0 ) {
			dherrno = dherror();

			switch ( dherrno ) {
			case Dh_alloc:
				UNIX_ERROR( "dhesearch() failed to allocate memory" );
				break;
			case Dh_collision:
				FPUT_SRC_CODE( stderr );
				(void) fputs( "insert in to hash table exceeded maximum number of collisions.\n", stderr );
				break;
			default:
				FPUT_SRC_CODE( stderr );
				(void) fputs( "programming error--unknown error (", stderr );
				(void) fput_int( dherrno, stderr );
				(void) fputs( ") occurred in dhesearch().\n", stderr );
				abort();
			}
			RETURN_INT( -1 );
		}

		if ( *bind_entry != bind_tbl ) {
			/*
			** duplicate field name
			*/
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Encountered duplicate field name [", stderr );
			(void) fputs( bind_tbl->field_name, stderr );
			(void) fputs( "].\n", stderr );
			RETURN_INT( -1 );
		}

		++bind_tbl;
	}

	*ret_hash_table = hash_table;

	RETURN_INT( 0 );
}

static unsigned long calc_hash( const void *x )
{
	return strhkey( ( (const dfile_bind_t *)x )->field_name );
}
