/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <zlib.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"

static const char       rcsid[] = "$Id: _dfile_check_zipped_crc.c,v 1.2 2009/10/16 07:35:53 keith Exp $";

/*
** $Log: _dfile_check_zipped_crc.c,v $
** Revision 1.2  2009/10/16 07:35:53  keith
** Add GPL to source code.
**
** Revision 1.1  2009/02/21 22:26:54  keith
** Initial revision
**
*/

static unsigned long parse_ulong( const unsigned char * );

/*
** This function compares the uncompressed 32-bit CRC and byte count
** read at the end of a zipped file with calculated CRC and byte count.
** It is a validation to be sure the zipped file is not corrupted.
*/

int _dfile_check_zipped_crc( dfile_t *dfile )
{
	static const char	func[] = "_dfile_check_zipped_crc";
	unsigned char	buf[8], *next_in;
	z_streamp	zstream;
	size_t	avail_in;
	unsigned long	crc, char_cnt;
	int	zerr;

	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	if ( _dfile_copy_raw_gz_data( buf, sizeof( buf ), dfile ) == -1 ) {
		RETURN_INT( -1 );
	}

	crc = parse_ulong( buf );
	char_cnt = parse_ulong( &buf[ 4 ] );

	if ( Debug ) {
		(void) fprintf( stderr, "Calculated CRC %lu, File CRC %lu\n", dfile->crc, crc );
		(void) fprintf( stderr, "Calculated character count %lu, File character count %lu\n", dfile->file_char_cnt, char_cnt );
	}

	if ( char_cnt != dfile->file_char_cnt ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Character count comparison failed for zipped file.\n", stderr );
		RETURN_INT( -1 );
	}

	if ( crc != dfile->crc ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "CRC comparison failed for zipped file.\n", stderr );
		RETURN_INT( -1 );
	}

	zstream = (z_stream *)dfile->zstream;
	assert( zstream != (z_stream *)0 );

	next_in = (unsigned char *)zstream->next_in;
	avail_in = (size_t)zstream->avail_in;

	zerr = inflateReset( zstream );
	if ( zerr != Z_OK ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "inflateReset() failed (", stderr );
		(void) fput_int( zerr, stderr );
		(void) fputs( "): ", stderr );
		if ( zstream->msg == (char *)0 ) {
			(void) fputs( "no description available of error", stderr );
		} else {
			(void) fputs( zstream->msg, stderr );
		}
		(void) fputs( ".\n", stderr );
		RETURN_INT( -1 );
	}

	zstream->next_in = (Bytef *)next_in;
	zstream->avail_in = (unsigned)avail_in;
	dfile->file_char_cnt = 0UL;
	dfile->crc = crc32( 0UL, (const Bytef *)0, 0U );

	RETURN_INT( 0 );
}

static unsigned long parse_ulong( const unsigned char *x )
{
	unsigned long	result;

	result = (unsigned long)x[ 0 ];
	result += (unsigned long)x[ 1 ] << 8;
	result += (unsigned long)x[ 2 ] << 16;
	result += (unsigned long)x[ 3 ] << 24;

	return result;
}
