/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_utility.h"

static const char       rcsid[] = "$Id: copy_record.c,v 1.2 2009/10/16 20:07:58 keith Exp $";

/*
** $Log: copy_record.c,v $
** Revision 1.2  2009/10/16 20:07:58  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/08 07:40:10  keith
** Initial revision
**
*/

/*
** This function copies a DFile record from its buffer into sorting memory.
*/
int copy_record( field_t **rec, size_t *rec_size, const dfile_bind_t **bind, unsigned short bind_cnt )
{
	static const char	func[] = "copy_record";
	size_t	record_size, field_length;
	unsigned short	ndx;
	char	*new_buffer, *buffer, *ptr;
	field_t	*field, *field_vector;
	const dfile_bind_t	**bind_ptr;

	assert( rec != (field_t **)0 );
	assert( rec_size != (size_t *)0 );
	assert( bind != (const dfile_bind_t **)0 );
	assert( bind_cnt > (unsigned short)0 );

	DEBUG_FUNC_START;

	/*
	** Calculate record size.
	*/
	record_size = (size_t)0;
	bind_ptr = bind;

	for ( ndx = bind_cnt; ndx > (unsigned short)0; --ndx ) {
		record_size += (size_t)*( *bind_ptr )->field_length;
		++bind_ptr;
	}

	/*
	** Add space for nulls.
	*/
	record_size += (size_t)bind_cnt;

	if ( Debug ) {
		(void) fprintf( stderr, "record_size = %u, previous allocated record size %u\n", record_size, *rec_size );
	}

	field_vector = *rec;

	if ( record_size > *rec_size ) {
		if ( field_vector == (field_t *)0 ) {
			/*
			** This should only occur on first call.
			*/
			if ( Debug ) {
				(void) fputs( "allocating field_vector\n", stderr );
			}
			field_vector = (field_t *)malloc( (size_t)bind_cnt * sizeof( field_t ) );
			if ( field_vector == (field_t *)0 ) {
				UNIX_ERROR( "malloc() failed" );
				RETURN_INT( -1 );
			}
			*rec = field_vector;
			buffer = (char *)0;
			if ( Debug ) {
				(void) fputs( "field_vector allocated\n", stderr );
			}
		} else {
			assert( field_vector != (field_t *)0 );
			assert( field_vector[ 0 ].value != (char *)0 );

			buffer = field_vector[ 0 ].value;
		}

		if ( Debug ) {
			(void) fputs( "expanding buffer\n", stderr );
		}

		new_buffer = (char *)realloc( (void *)buffer, record_size );

		if ( new_buffer == (char *)0 ) {
			UNIX_ERROR( "realloc() failed" );
			RETURN_INT( -1 );
		}

		buffer = new_buffer;
		*rec_size = record_size;

		if ( Debug ) {
			(void) fputs( "buffer expanded\n", stderr );
		}
	} else {
		assert( field_vector != (field_t *)0 );
		assert( field_vector[ 0 ].value != (char *)0 );

		buffer = field_vector[ 0 ].value;
	}

	ptr = buffer;
	field = field_vector;
	for ( ; bind_cnt > (unsigned short)0; --bind_cnt ) {
		field_length = *( *bind )->field_length;
		if ( Debug ) {
			(void) fprintf( stderr, "memcpy( %p, %p, %u )\n", ptr, *( *bind )->field_buffer, field_length );
		}
		if ( field_length > (size_t)0 ) {
			(void) memcpy( (void *)ptr, (void *)*( *bind )->field_buffer, field_length );
		}
		ptr[ field_length ] = (char)0;
		field->value = ptr;
		field->length = field_length;
		ptr += field_length + (size_t)1;
		++field;
		++bind;
		if ( Debug ) {
			(void) fputs( "copy complete\n", stderr );
		}
	}

	RETURN_INT( 0 );
}
