/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <search.h>
#include <unistd.h>
#include <assert.h>
#include "tbox.h"
#include "rbtree.h"
#include "dfile_exec.h"

static const char       rcsid[] = "$Id: wait_job.c,v 1.2 2009/10/16 20:05:35 keith Exp $";

/*
** $Log: wait_job.c,v $
** Revision 1.2  2009/10/16 20:05:35  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/07 08:27:06  keith
** Initial revision
**
*/

void rm_job_from_list( job_t **, job_t * );

/*
** This function waits for a job to complete, interprets job's exit code
** and performs post job cleanup.
*/
int wait_job( unsigned short *jobs_remaining_cnt, void **job_tree, int *exit_code, unsigned short *failure_cnt, job_t **job_list, unsigned short retries )
{
	static const char	func[] = "wait_job";
	pid_t	pid;
	int	status;
	char	msg[300], pid_str[24], int_str[24];
	size_t	len;
	job_t	key, **ptr, *job;

	assert( jobs_remaining_cnt != (unsigned short *)0 );
	assert( failure_cnt != (unsigned short *)0 );
	assert( job_tree != (void **)0 );
	assert( job_list != (job_t **)0 );
	assert( exit_code != (int *)0 );

	DEBUG_FUNC_START;

	/*
	** Wait for an application slice (job) to complete.
	*/
	pid = wait( &status );

	if ( pid < (pid_t) 0 ) {
		UNIX_ERROR( "wait() failed" );
		RETURN_INT( -1 );
	}

	if ( snprintf( pid_str, sizeof( pid_str ), "%d", pid ) <= 0 ) {
		UNIX_ERROR( "snprintf() failed to convert PID to ASCII" );
		RETURN_INT( -1 );
	}

	/*
	** Find which job completed using Unix process ID.
	*/
	key.pid = pid;
	ptr = (job_t **)rbtfind( (void *)&key, job_tree, jobcmp );
	if ( ptr == (job_t **)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to find PID ", stderr );
		(void) fputs( pid_str, stderr );
		(void) fputs( " in job list.\n", stderr );
		RETURN_INT( 0 );
	}

	job = *ptr;

	if ( WIFSIGNALED( status ) != 0 ) {
		/*
		** Child process death caused by signal.
		*/
		(void) strcpy( msg, get_ctime() );
		(void) strcat( msg, " PROCESS " );
		(void) strcat( msg, pid_str );
		(void) strcat( msg, " TERMINATED FROM RECEIVING SIGNAL " );
		if ( snprintf( int_str, sizeof( int_str ), "%d", WTERMSIG( status ) ) <= 0 ) {
			UNIX_ERROR( "snprintf() failed to convert signal value to ASCII" );
			RETURN_INT( -1 );
		}
		(void) strcat( msg, int_str );
		(void) strcat( msg, ".\n" );
		len = strlen( msg );
		if ( write( job->stderr_fd, (void *)msg, len ) != len ) {
			UNIX_ERROR( "write() failed on stderr file" );
		}
	}

	/*
	** Default status to failure.
	*/
	job->status = 1;

	if ( WIFEXITED( status ) != 0 ) {
		/*
		** Process terminated normally.
		*/
		job->status = WEXITSTATUS( status );

		(void) strcpy( msg, get_ctime() );
		(void) strcat( msg, " PROCESS " );
		(void) strcat( msg, pid_str );
		if ( WEXITSTATUS( status ) == 0 ) {
			(void) strcat( msg, " COMPLETED" );
		} else {
			(void) strcat( msg, " FAILED WITH RETURN CODE " );
			if ( snprintf( int_str, sizeof( int_str ), "%d", WEXITSTATUS( status ) ) <= 0 ) {
				UNIX_ERROR( "snprintf() failed to convert return value to ASCII" );
				RETURN_INT( -1 );
			}
			(void) strcat( msg, int_str );
		}
		(void) strcat( msg, ".\n" );
		len = strlen( msg );

		if ( job->stderr_fd != 2 ) {
			/*
			** Completed job did not inherit parent's stderr.
			*/
			if ( write( job->stderr_fd, (void *)msg, len ) != len ) {
				UNIX_ERROR( "write() failed on stderr file" );
			}
		}
	}

	/*
	** Close parent's copy of stdout and stderr file descriptors used
	** by completed job.
	*/
	if ( job->stdout_fd != 1 ) {
		/*
		** Completed job did not inherit parent's stdout.
		*/
		if ( close( job->stdout_fd ) == -1 ) {
			UNIX_ERROR( "close() failed on stdout file" );
		}
	}

	if ( job->stderr_fd != 2 ) {
		/*
		** Completed job did not inherit parent's stderr.
		*/
		if ( close( job->stderr_fd ) == -1 ) {
			UNIX_ERROR( "close() failed on stderr file" );
		}
	}

	++job->attempts;

	if ( job->status == 0 ) {
		/*
		** Job completed successfully.
		*/
		--*jobs_remaining_cnt;

		/*
		** Remove job from run list.
		*/
		rm_job_from_list( job_list, job );
	} else {
		/*
		** Job did not complete successfully.
		*/
		if ( job->attempts > retries ) {
			/*
			** Exceeded maximum number of reties.
			*/
			--*jobs_remaining_cnt;

			/*
			** Remove job from run list.
			*/
			rm_job_from_list( job_list, job );

			++*failure_cnt;

			/*
			** At least one process did not complete successfully.
			** Eventually exit with a failure return code.
			*/
			*exit_code = 100;
		}
	}

	/*
	** Remove process ID from tree.
	*/
	ptr = (job_t **)rbtdelete( (void *)&key, job_tree, jobcmp );
	if ( ptr == (job_t **)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to delete PID ", stderr );
		(void) fputs( pid_str, stderr );
		(void) fputs( " from job tree.\n", stderr );
	}

	(void) fputs( get_ctime(), stderr );
	(void) fputs( " Process ", stderr );
	(void) fputs( pid_str, stderr );
	(void) fputs( " completed (", stderr );
	(void) fprintf( stderr, "%d", job->status );
	(void) fputs( ").\n", stderr );
	

	RETURN_INT( 0 );
}

void rm_job_from_list( job_t **job_list, job_t *job )
{
	if ( *job_list == job ) {
		*job_list = job->next;
		if ( *job_list != (job_t *)0 ) {
			( *job_list )->prev = (job_t *)0;
		}
	} else {
		if ( job->next != (job_t *)0 ) {
			job->next->prev = job->prev;
		}
		job->prev->next = job->next;
	}
}
