/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <search.h>
#include <assert.h>
#include "tbox.h"
#include "rbtree.h"
#include "dfile_exec.h"

static const char       rcsid[] = "$Id: start_jobs.c,v 1.2 2009/10/16 20:05:35 keith Exp $";

/*
** $Log: start_jobs.c,v $
** Revision 1.2  2009/10/16 20:05:35  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/07 08:10:08  keith
** Initial revision
**
*/

/*
** This function starts as many available application jobs (slices) as
** possible without exceeding maximum concurrent jobs.
*/
int start_jobs( unsigned short *jobs_running_cnt, unsigned short *allocated_job_cnt, job_t *job_list, void **job_tree, const char **divvy_text, unsigned short divvy_text_cnt, const char * const *input_fname, unsigned short input_fname_cnt, slice_method_t slice_method_flag, unsigned short max_concurrent_jobs, unsigned short retries, const char *output_name, int semid, unsigned short jobs_remaining_cnt )
{
	static const char	func[] = "start_jobs";
	const char	*stdout_fname, *stderr_fname, *input_file;
	pid_t	pid;
	int	stdout_fd, stderr_fd;
	job_t	**ptr;
	unsigned short	job_cnt;
	int	resource_flag;
	const char	*slice_text;

	assert( jobs_running_cnt != (unsigned short *)0 );
	assert( job_list != (job_t *)0 );
	assert( job_tree != (void **)0 );
	assert( slice_method_flag == File_slice || slice_method_flag == Parm_file_slice );
	/*
	** output_name may be null.
	** divvy_text will be null when slice_method_flag is File_slice.
	*/

	DEBUG_FUNC_START;

	if ( get_job_cnt( &job_cnt, divvy_text_cnt, input_fname_cnt, slice_method_flag ) == -1 ) {
		RETURN_INT( -1 );
	}

	/*
	** Cycle through job table looking for jobs to start.
	*/
	for ( ; job_list != (job_t *)0; job_list = job_list->next ) {
		if ( *jobs_running_cnt >= max_concurrent_jobs ) {
			/*
			** Don't start any more jobs.
			*/
			break;
		}

		if ( Debug ) {
			fprintf( stderr, "*allocated_job_cnt = %hu, max_concurrent_jobs = %hu, jobs_remaining_cnt = %hu, *jobs_running_cnt = %hu\n", *allocated_job_cnt, max_concurrent_jobs, jobs_remaining_cnt, *jobs_running_cnt );
		}

		if ( semid >= 0 && max_concurrent_jobs > *allocated_job_cnt && jobs_remaining_cnt > *allocated_job_cnt && *jobs_running_cnt >= *allocated_job_cnt ) {
			if ( request_job_resource( &resource_flag, *allocated_job_cnt, semid ) == -1 ) {
				RETURN_INT( -1 );
			}
			if ( resource_flag == 0 ) {
				/*
				** Could not get another resource.
				*/
				break;
			}
			++*allocated_job_cnt;
		}

		if ( job_list->pid != (pid_t)0 ) {
			/*
			** Job previously started.
			*/
			if ( job_list->status <= 0 || job_list->attempts > retries ) {
				/*
				** Exceeded number of retries for failure,
				** job currently running, or
				** job completed successfully.
				*/
				continue;
			}
		}

		/*
		** Job ran and failed, retry failed,
		** or start job for first attempt.
		*/

		if ( slice_method_flag == Parm_file_slice ) {
			slice_text = divvy_text[ job_list->job_nbr ];
		} else {
			slice_text = (const char *)0;
		}

		/*
		** Assign file names to capture stdout and stderr of
		** application about to be started.
		*/
		if ( assign_fname( &stdout_fname, &stderr_fname, output_name, job_list->job_nbr, job_cnt, slice_text ) == -1 ) {

			RETURN_INT( -1 );
		}

		input_file = ( slice_method_flag == File_slice ) ? input_fname[ job_list->job_nbr ] : "/dev/null";

		/*
		** Start new instance of application (job) to process a slice.
		*/
		pid = exec_job( job_list->exec_args, stdout_fname, stderr_fname, &stdout_fd, &stderr_fd, input_file );
		if ( pid == (pid_t)-1 ) {
			RETURN_INT( -1 );
		}

		/*
		** Save job information.
		*/
		job_list->stdout_fd = stdout_fd;
		job_list->stderr_fd = stderr_fd;
		job_list->pid = pid;
		job_list->status = -1;

		/*
		** Record job in tree for Unix process ID lookups.
		*/
		ptr = (job_t **)rbtsearch( (void *)job_list, job_tree, jobcmp );
		if ( ptr == (job_t **)0 ) {
			/*
			** Out of memory.
			*/
			RETURN_INT( -1 );
		}

		if ( *ptr != job_list ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Duplicate process id ", stderr );
			(void) fprintf( stderr, "%d", pid );
			(void) fputs( " encountered.\n", stderr );
			RETURN_INT( -1 );
		}

		/*
		** Job successfully started and recorded.
		*/
		++*jobs_running_cnt;

		(void) fputs( get_ctime(), stderr );
		(void) fputs( " Started process ", stderr );
		(void) fprintf( stderr, "%d", pid );
		(void) fputs( " for slice ", stderr );
		(void) fprintf( stderr, "%hu", job_list->job_nbr );
		if ( slice_text != (const char *)0 ) {
			(void) fputs( " [", stderr );
			(void) fputs( slice_text, stderr );
			(void) fputs( "]", stderr );
		}
		(void) fputs( ".\n", stderr );
	}

	RETURN_INT( 0 );
}
