/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <tbox.h>
#include "dfile_exec.h"

static const char       rcsid[] = "$Id: main.c,v 1.2 2009/10/16 20:05:35 keith Exp $";

/*
** $Log: main.c,v $
** Revision 1.2  2009/10/16 20:05:35  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/07 04:51:20  keith
** Initial revision
**
*/

/*
** This program is used to run groups of data files concurrently.
*/
int main( int argc, char **argv )
{
	static const char	func[] = "main";
	static const char	End_msg[] = "End %s() returning %d at %s\n";
	slice_method_t	slice_method_flag;
	unsigned short	max_concurrent_jobs, retries;
	unsigned short	jobs_remaining_cnt, jobs_running_cnt, input_fname_cnt;
	unsigned short	divvy_text_cnt, max_failure_cnt, failure_cnt;
	unsigned short	allocated_job_cnt;
	unsigned short	unneeded_resource_cnt;
	const char	*command, *output_name, *input_file_pattern;
	const char	**input_fname, *divvy_fname, **divvy_text;
	const char	*recovery_log, *cpu_semid_fname;
	int	exit_code, cpu_semid;
	job_t	*job_tbl, *job_list;
	void 	*job_tree;

	/*
	** Process command line arguments.
	*/
	if ( get_args( argc, argv, &divvy_fname, &max_concurrent_jobs, &retries, &command, &output_name, &input_file_pattern, &recovery_log, &max_failure_cnt, &cpu_semid_fname ) == -1 ) {
		return 10;
	}

	(void) fprintf( stderr, Start_func_msg, func, get_ctime() );

	cpu_semid = get_cpu_semid( cpu_semid_fname );

	if ( divvy_fname != (const char *)0 ) {
		if ( get_divvy_text( &divvy_text, &divvy_text_cnt, divvy_fname ) == -1 ) {
			(void) fprintf( stderr, End_msg, func, 22, get_ctime() );
			return 22;
		}
	}

	slice_method_flag = get_slice_method( divvy_text_cnt, input_file_pattern );

	if ( slice_method_flag == Invalid_slice_method ) {
		(void) fprintf( stderr, End_msg, func, 25, get_ctime() );
		return 25;
	}


	switch ( slice_method_flag ) {
	case File_slice:
		if ( assign_input_fname( &input_fname, &input_fname_cnt, input_file_pattern ) == -1 ) {
			(void) fprintf( stderr, End_msg, func, 45, get_ctime() );
			return 45;
		}
		jobs_remaining_cnt = input_fname_cnt;
		break;
	case Parm_file_slice:
		jobs_remaining_cnt = divvy_text_cnt;
		break;
	}

	/*
	** This function allocates job table array. Its size corresponds to
	** number of slices to be processed.
	*/
	job_tbl = alloc_job_tbl( jobs_remaining_cnt );
	if ( job_tbl == (job_t *)0 ) {
		(void) fprintf( stderr, End_msg, func, 50, get_ctime() );
		return 50;
	}

	if ( assign_exec_args( command, job_tbl, divvy_text, divvy_text_cnt, input_fname_cnt, slice_method_flag ) == -1 ) {
		(void) fprintf( stderr, End_msg, func, 52, get_ctime() );
		return 52;
	}

	if ( load_recovery_log( job_tbl, &jobs_remaining_cnt, divvy_text_cnt, input_fname_cnt, slice_method_flag, recovery_log ) == -1 ) {
		(void) fprintf( stderr, End_msg, func, 55, get_ctime() );
		return 55;
	}

	if ( set_job_list( &job_list, job_tbl, divvy_text_cnt, input_fname_cnt, slice_method_flag ) == -1 ) {
		(void) fprintf( stderr, End_msg, func, 57, get_ctime() );
		return 57;
	}

	jobs_running_cnt = (unsigned short)0;
	allocated_job_cnt = (unsigned short)0;
	job_tree = (void *)0;
	exit_code = 0;
	failure_cnt = (unsigned short)0;

	/*
	** Loop until all slices have been processes.
	*/
	while ( jobs_remaining_cnt > (unsigned short)0 ) {
		if ( max_failure_cnt >= failure_cnt ) {
			/*
			** Start as many available application jobs (slices) as
			** possible without exceeding maximum concurrent jobs.
			*/
			if ( start_jobs( &jobs_running_cnt, &allocated_job_cnt, job_list, &job_tree, divvy_text, divvy_text_cnt, input_fname, input_fname_cnt, slice_method_flag, max_concurrent_jobs, retries, output_name, cpu_semid, jobs_remaining_cnt ) == -1 ) {
				(void) fprintf( stderr, End_msg, func, 60, get_ctime() );
				return 60;
			}
		}

		/*
		** Wait for a job to complete, interpret job's exit code
		** and perform post job cleanup.
		*/
		if ( wait_job( &jobs_remaining_cnt, &job_tree, &exit_code, &failure_cnt, &job_list, retries ) == -1 ) {
			(void) fprintf( stderr, End_msg, func, 70, get_ctime() );
			return 70;
		}

		if ( Debug ) {
			fprintf( stderr, "allocated_job_cnt = %hu, jobs_remaining_cnt = %hu\n", allocated_job_cnt, jobs_remaining_cnt );
		}

		if ( allocated_job_cnt > jobs_remaining_cnt ) {
			unneeded_resource_cnt = allocated_job_cnt - jobs_remaining_cnt;
			(void) release_allocated_jobs( unneeded_resource_cnt, cpu_semid );
			allocated_job_cnt -= unneeded_resource_cnt;
		}

		if ( Debug ) {
			(void) fprintf( stderr, "jobs_remaining_cnt = %hu\n", jobs_remaining_cnt );
		}

		--jobs_running_cnt;

		if ( failure_cnt > max_failure_cnt && jobs_running_cnt == (unsigned short)0 ) {
			break;
		}
	}

	if ( exit_code != 0 ) {
		(void) write_recovery_log( job_tbl, divvy_text_cnt, input_fname_cnt, slice_method_flag, recovery_log );
	}

	write_system_usage();

	(void) fprintf( stderr, End_msg, func, exit_code, get_ctime() );

	return exit_code;
}
