/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile_exec.h"

static const char       rcsid[] = "$Id: load_recovery_log.c,v 1.2 2009/10/16 20:05:35 keith Exp $";

/*
** $Log: load_recovery_log.c,v $
** Revision 1.2  2009/10/16 20:05:35  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/07 04:34:08  keith
** Initial revision
**
*/

/*
** This function loads a recovery log so slices will not be re-run.
*/
int load_recovery_log( job_t *job_tbl, unsigned short *jobs_remaining_cnt, unsigned short range_tbl_cnt, unsigned short input_fname_cnt, slice_method_t slice_method_flag, const char *recovery_log )
{
	static const char	func[] = "load_recovery_log";
	job_t	*job_ptr;
	unsigned short	job_cnt;
	FILE	*fptr;
	char	*int_ptr, buf[24];
	static const char	msg1[] = "Recovery file [";
	static const char	msg2[] = "] does not contain a slice number per line.\n";
	long	slice;
	size_t	len;

	assert( job_tbl != (job_t *)0 );
	assert( jobs_remaining_cnt != (unsigned short *)0 );
	assert( slice_method_flag == File_slice || slice_method_flag == Parm_file_slice );

	DEBUG_FUNC_START;

	if ( recovery_log == (const char *)0 ) {
		/*
		** Not using recovery log.
		*/
		RETURN_INT( 0 );
	}

	if ( access( recovery_log, F_OK ) == -1 ) {
		if ( errno == ENOENT ) {
			/*
			** Recovery log does not exist.
			*/
			RETURN_INT( 0 );
		}
	}

	if ( get_job_cnt( &job_cnt, range_tbl_cnt, input_fname_cnt, slice_method_flag ) == -1 ) {
		RETURN_INT( -1 );
	}

	fptr = fopen( recovery_log, "r" );

	if ( fptr == (FILE *)0 ) {
		UNIX_ERROR( "fopen() failed" );
		(void) fputs( "Could not open recovery file [", stderr );
		(void) fputs( recovery_log, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( -1 );
	}

	while ( fgets( buf, sizeof( buf ), fptr ) != (char *)0 ) {
		len = strlen( buf );
		if ( len == 0 ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "fgets() returned a zero length string.\n", stderr );
			RETURN_INT( -1 );
		}

		--len;

		if ( buf[ len ] != '\n' ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( msg1, stderr );
			(void) fputs( recovery_log, stderr );
			(void) fputs( msg2, stderr );
			RETURN_INT( -1 );
		}

		buf[ len ] = (char)0;
		slice = strtol( buf, &int_ptr, 10 );
		if ( int_ptr - buf != len ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( msg1, stderr );
			(void) fputs( recovery_log, stderr );
			(void) fputs( msg2, stderr );
			RETURN_INT( -1 );
		}

		if ( slice < 0L || slice > (long)job_cnt ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Slice number [", stderr );
			(void) fput_long( slice, stderr );
			(void) fputs( "] is not between 0 and ", stderr );
			(void) fput_ushort( job_cnt, stderr );
			(void) fputs( ".\n", stderr );
			RETURN_INT( -1 );
		}

		if ( *jobs_remaining_cnt == (unsigned short)0 ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Recovery file [", stderr );
			(void) fputs( recovery_log, stderr );
			(void) fputs( "] contained more completed jobs than actual jobs to run.\n", stderr );
			RETURN_INT( -1 );
		}

		--*jobs_remaining_cnt;

		/*
		** Create fake entry.
		*/
		job_ptr = &job_tbl[ slice ];
		job_ptr->pid = (pid_t)1;
		job_ptr->status = 0;
		job_ptr->attempts = (unsigned short)0;
	}

	if ( ferror( fptr ) ) {
		UNIX_ERROR( "fgets() failed" );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to read recovery file [", stderr );
		(void) fputs( recovery_log, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( -1 );
	}

	if ( fclose( fptr ) == EOF ) {
		UNIX_ERROR( "fclose() failed" );
	}

	if ( unlink( recovery_log ) == -1 ) {
		UNIX_ERROR( "unlink() failed" );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to remove recovery log [", stderr );
		(void) fputs( recovery_log, stderr );
		(void) fputs( "].\n", stderr );

		RETURN_INT( -1 );
	}

	RETURN_INT( 0 );
}
