/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <search.h>
#include <assert.h>
#include "tbox.h"
#include "rbtree.h"
#include "dfile_exec.h"

static const char       rcsid[] = "$Id: assign_exec_args.c,v 1.2 2009/10/16 20:05:35 keith Exp $";

/*
** $Log: assign_exec_args.c,v $
** Revision 1.2  2009/10/16 20:05:35  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/06 10:15:42  keith
** Initial revision
**
*/

/*
** This function builds and parses commands to execute.
*/
int assign_exec_args( const char *command, job_t *job_tbl, const char **divvy_text, unsigned short divvy_text_cnt, unsigned short input_fname_cnt, slice_method_t slice_method_flag )
{
	static const char	func[] = "assign_exec_args";
	unsigned short	job_ndx;
	const char	*cmd;
	unsigned short	job_cnt;
	const char	*slice_name;
	char	sequence[24], new_cmd[24*1024];

	assert( command != (const char *)0 );
	assert( job_tbl != (job_t *)0 );
	assert( slice_method_flag == File_slice || slice_method_flag == Parm_file_slice );
	/*
	** output_name may be null.
	** divvy_text will be null when slice_method_flag is File_slice.
	*/

	DEBUG_FUNC_START;

	if ( get_job_cnt( &job_cnt, divvy_text_cnt, input_fname_cnt, slice_method_flag ) == -1 ) {
		RETURN_INT( -1 );
	}

	/*
	** Make sure command isn't too large to process.
	*/
	if ( strlen( command ) >= sizeof( new_cmd ) ) {
		(void) fputs( __FILE__, stderr );
		(void) fprintf( stderr, "(%d)", __LINE__ );
		(void) fputs( ": Command [", stderr );
		(void) fputs( command, stderr );
		(void) fputs( "] exceeds maximum length ", stderr );
		(void) fprintf( stderr, "%u", sizeof( new_cmd ) );
		(void) fputs( ".\n", stderr );
		RETURN_INT( -1 );
	}

	for ( job_ndx = (unsigned short)0; job_ndx < job_cnt; ++job_ndx ) {
		job_tbl[ job_ndx ].job_nbr = job_ndx;

		if ( slice_method_flag == Parm_file_slice ) {
			assert( divvy_text_cnt > job_ndx );
			slice_name = divvy_text[ job_ndx ];
		} else {
			slice_name = (const char *)0;
		}

		/*
		** Search command line arguments for %n and %s flags.
		** These flags are replaced with slice sequence number
		** and partition name respectively.
		*/
		if ( snprintf( sequence, sizeof( sequence ), "%hu", job_ndx ) <= 0 ) {
			UNIX_ERROR( "sprintf() failed to convert job_ndx to ASCII" );
			RETURN_INT( -1 );
		}

		(void) strcpy( new_cmd, command );

		/*
		** Replace %n and %s flags that occur in command.
		*/
		if ( substitute_tokens( new_cmd, sizeof( new_cmd ), sequence, slice_name ) == -1 ) {
			RETURN_INT( -1 );
		}

		cmd = strenv( new_cmd );
		if ( cmd == (const char *)0 ) {
			RETURN_INT( -1 );
		}

		/*
		** Make sure command isn't too large to process.
		*/
		if ( strlen( cmd ) >= sizeof( new_cmd ) ) {
			(void) fputs( __FILE__, stderr );
			(void) fprintf( stderr, "(%d)", __LINE__ );
			(void) fputs( ": Command [", stderr );
			(void) fputs( cmd, stderr );
			(void) fputs( "] exceeds maximum length ", stderr );
			(void) fprintf( stderr, "%u", sizeof( new_cmd ) );
			(void) fputs( ".\n", stderr );
			RETURN_INT( -1 );
		}

		(void) strcpy( new_cmd, cmd );

		/*
		** Parse command line arguments of application to be run.
		** Results will later be used in execvp() call.
		*/
		if ( parse_exec_args( new_cmd, &job_tbl[ job_ndx ].exec_args ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	RETURN_INT( 0 );
}
