/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "dfile_cache_create.h"

static const char       rcsid[] = "$Id: get_args.c,v 1.3 2009/10/16 19:49:58 keith Exp $";

/*
** $Log: get_args.c,v $
** Revision 1.3  2009/10/16 19:49:58  keith
** Added GPL to source code.
**
** Revision 1.2  2009/03/18 00:31:03  keith
** Added checks for NDEBUG macro.
**
** Revision 1.1  2009/03/04 16:36:48  keith
** Initial revision
**
*/

#define NULL_CHECK( x ) ( ( *x == (const char *)0 ) ? "null" : *x )

static void	print_usage( const char * );

/*
** This function processes the command line arguments.
*/
int get_args( int argc, char * const argv[], const char **input_dfile_name, dfile_tag_t **dfile_tag_tbl, unsigned short *dfile_tag_tbl_cnt, const char **ipc_key, const char **input_filter_file_name )
{
	int	ch, ndx;
	extern char	*optarg;

	assert( argv != (char * const *) 0 );
	assert( input_dfile_name != (const char **) 0 );
	assert( dfile_tag_tbl != (dfile_tag_t **) 0 );
	assert( dfile_tag_tbl_cnt != (unsigned short *) 0 );
	assert( ipc_key != (const char **)0 );
	assert( input_filter_file_name != (const char **)0 );

#ifndef NDEBUG
	/*
	** Initialize debug to off.
	*/
	Debug = 0;
#endif

	*input_dfile_name = (const char *)0;
	*dfile_tag_tbl = (dfile_tag_t *)0;
	*dfile_tag_tbl_cnt = (unsigned short)0;
	*ipc_key = (const char *)0;
	*input_filter_file_name = (const char *)0;

	while ( ( ch = getopt( argc, argv, "a:di:t:y:" ) ) != EOF ) {
		switch ( ch ) {
		case 'a':
			*ipc_key = optarg;
			break;
		case 'd':
#ifdef NDEBUG
			(void) fputs( "Not compiled for debug.\n", stderr );
#else
			Debug = 1;
#endif
			break;
		case 'i':
			*input_dfile_name = optarg;
			break;
		case 't':
			if ( cmd_line_parse_tag( dfile_tag_tbl, dfile_tag_tbl_cnt, optarg ) == -1 ) {
				return -1;
			}
			break;
		case 'y':
			*input_filter_file_name = optarg;
			break;
		default:
			print_usage( argv[ 0 ] );
			return -1;
		}
	}

	if ( *input_dfile_name == (char *) 0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify input dfile name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *ipc_key == (char *) 0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify IPC key.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, "Input DFile Name: [%s]\n", NULL_CHECK( input_dfile_name ) );
		(void) fprintf( stderr, "Input Filter File Name: [%s]\n", NULL_CHECK( input_filter_file_name ) );
		(void) fprintf( stderr, "IPC Key: [%s]\n", *ipc_key );
		for ( ndx = (unsigned short)0; ndx < *dfile_tag_tbl_cnt; ++ndx ) {
			(void) fprintf( stderr, "DFile Tag [%s] [%s]\n", ( *dfile_tag_tbl )[ ndx ].tag, ( *dfile_tag_tbl )[ ndx ].tag_value );
		}
	}

	return 0;
}

static void print_usage( const char *exec_name )
{
	(void) fputs( "usage: ", stderr );
	(void) fputs( exec_name, stderr );
	(void) fputs( " -a 0xN", stderr );
	(void) fputs( " [-d]", stderr );
	(void) fputs( " -i dfile", stderr );
	(void) fputs( " [-t %x=abc]", stderr );
	(void) fputs( " [-y file]", stderr );
	(void) fputc( '\n', stderr );
	(void) fputs( "\t-a -> IPC key for shared memory\n", stderr );
	(void) fputs( "\t-d -> debug\n", stderr );
	(void) fputs( "\t-i -> input DFile\n", stderr );
	(void) fputs( "\t-t -> DFile path tags\n", stderr );
	(void) fputs( "\t-y -> join filter file\n", stderr );
}
