/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_agfunc.h"

static const char       rcsid[] = "$Id: process_agfunc.c,v 1.2 2009/10/16 19:43:49 keith Exp $";

/*
** $Log: process_agfunc.c,v $
** Revision 1.2  2009/10/16 19:43:49  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/02 05:19:13  keith
** Initial revision
**
*/

static int convert_value( double *, const char * );
static int new_str( char **, const char * );
static int retain_orig_str( char *, size_t, const char *, size_t, const char * );

/*
** This function is called for each record to evaluate an aggregate function.
*/
int process_agfunc( func_t *func_ptr )
{
	static const char	func[] = "process_agfunc";
	double	numeric_value;

	assert( func_ptr != (func_t *)0 );

	DEBUG_FUNC_START;

	switch ( func_ptr->func_type ) {
	case Average:
	case Sum:
		if ( convert_value( &numeric_value, *func_ptr->src_bind->field_buffer ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( func_ptr->initialize_flag ) {
			/*
			** Set initialize flag to false.
			*/
			func_ptr->initialize_flag = 0;
			func_ptr->value_u.num = numeric_value;
		} else {
			func_ptr->value_u.num += numeric_value;
		}
		break;

	case Num_min:
		if ( convert_value( &numeric_value, *func_ptr->src_bind->field_buffer ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( func_ptr->initialize_flag ) {
			/*
			** Set initialize flag to false.
			*/
			func_ptr->initialize_flag = 0;
			func_ptr->value_u.num = numeric_value;

			if ( retain_orig_str( func_ptr->format_result, sizeof( func_ptr->format_result ), *func_ptr->src_bind->field_buffer, *func_ptr->src_bind->field_length, "minimum" ) == -1 ) {
				RETURN_INT( -1 );
			}
		} else {
			if ( func_ptr->value_u.num > numeric_value ) {
				func_ptr->value_u.num = numeric_value;

				if ( retain_orig_str( func_ptr->format_result, sizeof( func_ptr->format_result ), *func_ptr->src_bind->field_buffer, *func_ptr->src_bind->field_length, "minimum" ) == -1 ) {
					RETURN_INT( -1 );
				}
			}
		}
		break;

	case Num_max:
		if ( convert_value( &numeric_value, *func_ptr->src_bind->field_buffer ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( func_ptr->initialize_flag ) {
			/*
			** Set initialize flag to false.
			*/
			func_ptr->initialize_flag = 0;
			func_ptr->value_u.num = numeric_value;

			if ( retain_orig_str( func_ptr->format_result, sizeof( func_ptr->format_result ), *func_ptr->src_bind->field_buffer, *func_ptr->src_bind->field_length, "maximum" ) == -1 ) {
				RETURN_INT( -1 );
			}
		} else {
			if ( numeric_value > func_ptr->value_u.num ) {
				func_ptr->value_u.num = numeric_value;

				if ( retain_orig_str( func_ptr->format_result, sizeof( func_ptr->format_result ), *func_ptr->src_bind->field_buffer, *func_ptr->src_bind->field_length, "maximum" ) == -1 ) {
					RETURN_INT( -1 );
				}
			}
		}
		break;

	case Str_max:
		if ( func_ptr->initialize_flag ) {
			/*
			** Set initialize flag to false.
			*/
			func_ptr->initialize_flag = 0;
			if ( new_str( &func_ptr->value_u.str, *func_ptr->src_bind->field_buffer ) == -1 ) {
				RETURN_INT( -1 );
			}
		} else {
			if ( strcmp( *func_ptr->src_bind->field_buffer, func_ptr->value_u.str ) > 0 ) {
				free( (void *)func_ptr->value_u.str );
				if ( new_str( &func_ptr->value_u.str, *func_ptr->src_bind->field_buffer ) == -1 ) {
					RETURN_INT( -1 );
				}
			}
		}
		break;

	case Str_min:
		if ( func_ptr->initialize_flag ) {
			/*
			** Set initialize flag to false.
			*/
			func_ptr->initialize_flag = 0;
			if ( new_str( &func_ptr->value_u.str, *func_ptr->src_bind->field_buffer ) == -1 ) {
				RETURN_INT( -1 );
			}
		} else {
			if ( strcmp( func_ptr->value_u.str, *func_ptr->src_bind->field_buffer ) > 0 ) {
				free( (void *)func_ptr->value_u.str );
				if ( new_str( &func_ptr->value_u.str, *func_ptr->src_bind->field_buffer ) == -1 ) {
					RETURN_INT( -1 );
				}
			}
		}
		break;

	case Count:
		/*
		** do nothing
		*/
		break;

	default:
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Programming error--unknown func_type [", stderr );
		(void) fput_int( func_ptr->func_type, stderr );
		(void) fputs( "] encountered.\n", stderr );
		abort();
	}

	RETURN_INT( 0 );
}

static int convert_value( double *result, const char *str )
{
	char	*end_ptr;

	if ( *str == (char)0 ) {
		/*
		** zero length string
		*/
		*result = 0.0;
		return 0;
	}

	*result = strtod( str, &end_ptr );

	if ( end_ptr == str || *end_ptr != (char)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "strtod() failed to convert [", stderr );
		(void) fputs( str, stderr );
		(void) fputs( "].\n", stderr );
		return -1;
	}

	return 0;
}

static int new_str( char **result, const char *str )
{
	*result = strdup( str );
	if ( *result == (char *)0 ) {
		UNIX_ERROR( "strdup() failed" );
		return -1;
	}

	return 0;
}

static int retain_orig_str( char *dest, size_t dest_size, const char *value, size_t value_len, const char *desc )
{
	if ( value_len >= dest_size ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( desc, stderr );
		(void) fputs( " value [", stderr );
		(void) fputs( value, stderr );
		(void) fputs( "] exceeded maximum length ", stderr );
		(void) fput_uint( dest_size-1, stderr );
		(void) fputs( ".\n", stderr );

		return -1;
	}

	(void) memcpy( (void *)dest, (void *)value, value_len );
	dest[ value_len ] = (char)0;

	return 0;
}
