"""basic cfvers classes

This module implements the basic cfvers objects, in a
repository-independent way.

"""

# Copyright 2003 Iustin Pop
#
# This file is part of cfvers.
#
# cfvers is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# cfvers is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with cfvers; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os, struct, stat, os.path, re, commands, sys
import base64
import types
import random
import difflib
import time
import bz2
from cStringIO import StringIO
from mx import DateTime

__all__ = ["Area", "Item", "RevEntry", "AreaRev", "forcejoin"]

def forcejoin(a, *p):
    """Join two or more pathname components, considering them all relative"""
    path = a
    for b in p:
        if path[-1:] == '/' and b[:1] == '/':
            path = path + b[1:]
        elif path[-1:] != '/' and b[:1] != '/':
            path = path + '/' + b
        else:
            path = path + b
    return path
            
class Area(object):
    """Class implementing an area object.

    An area is a independent group of versioned items in a
    repository. You will usually use more than one area for different
    servers in a networked repository, and/or for multiple chroot
    jails on one server.
    
    """
    __slots__ = ["server", "id", "revno", "_ctime", "name", "root", "description", "numitems"]
    
    def __init__(self, id=None, server=None, name="default",
                 description="", root="/", revno=0, ctime = None,
                 numitems=0):
        """Constructor for the Area class"""
        if server is None:
            self.server = os.uname()[1]
        else:
            self.server = server
        self.id = id
        self.revno = revno
        self.ctime = ctime or DateTime.utc()
        self.name = name
        self.description = description
        self.root = root
        self.numitems = numitems

    def _set_ctime(self, val):
        if isinstance(val, types.StringTypes):
            val = DateTime.ISO.ParseDateTimeUTC(val)
        self._ctime = val

    def _get_ctime(self):
        return self._ctime

    ctime = property(_get_ctime, _set_ctime, None,
                     "The creation time of this area")
    
    def parse(repo, value):
        """Converts a string representation to an server, area object pair"""
        try:
            areaid = int(value)
        except ValueError:
            areaid = None
        if areaid is not None:
            return repo.getArea(areaid)
        m = re.match("^((?P<server>.*?):)?(?P<area>.*)$", value)
        if m is None:
            raise ValueError, "Invalid area specification: %s" % value
        server = m.group('server')
        if server is None:
            server = os.uname()[1]
        return repo.getAreaByName(server, m.group('area'))
    parse = staticmethod(parse)


class Item(object):
    __slots__ = ["id", "area", "name", "ctime"]
    
    def __init__(self, id=-1, area=None, name=None, ctime=None):
        if name is None or not name.startswith("/"):
            raise ValueError, "Invalid name '%s'" % name
        self.id = id
        self.area = area
        self.name = name
        if ctime is None:
            self.ctime = DateTime.utc()
        elif not isinstance(ctime, DateTime.DateTimeType):
            self.ctime = DateTime.ISO.ParseDateTimeUTC(ctime)
        else:
            self.ctime = ctime
        return

    def __str__(self):
        return "Item: id #%d, name %s" % (self.id, self.name)

class RevEntry(object):
    __slots__ = ["item", "revno", "filename", "filetype", "filecontents",
                 "mode", "mtime", "atime", "uid", "gid", "rdev",
                 "encoding",]
                 
    def __init__(self, vi = None, revno = None):
        if vi is None:
            # the init will be done by hand
            return
        source = forcejoin(vi.area.root, vi.name)
        self.filename = vi.name
        self.item = vi.id
        self.revno = revno
        st = os.lstat(source)

        self.filetype = stat.S_IFMT(st.st_mode)
        if self.filetype == stat.S_IFREG:
            self.filecontents = file(source, "r").read()
        elif self.filetype == stat.S_IFDIR:
            self.filecontents = "\n".join(os.listdir(source))
        elif self.filetype == stat.S_IFLNK:
            self.filecontents = os.readlink(source)
        else:
            self.filecontents = ""
        self.mode = st.st_mode
        self.mtime = st.st_mtime
        self.atime = st.st_atime
        self.uid = st.st_uid
        self.gid = st.st_gid
        self.rdev = st.st_rdev
        return

    def _mknod(path, mode, device):
        if hasattr(os, mknod):
            of.mknod(path, mode, device)
        else:
            if mode & stat.S_IFCHR:
                os.system("mknod '%s' c %u %u" % (path, self.rdev >> 8, self.rdev & 255))
            elif mode & stat.S_IFBLK:
                os.system("mknod '%s' b %u %u" % (newname, self.rdev >> 8, self.rdev & 255))
            else:
                raise TypeError, "Unknown mode passed to mknod!"
        return

    _mknod = staticmethod(_mknod)

    def _diffdata(older, newer, ofname=None, nfname=None, oftime=None, nftime=None):
        if hasattr(difflib, 'unified_diff'):
            a = older.splitlines(True)
            b = newer.splitlines(True)
            differ = difflib.unified_diff(
                a, b,
                ofname, nfname,
                oftime, nftime,
                )
            data = "".join(differ)
        else:
            f1 = file("/tmp/a", "w")
            f2 = file("/tmp/b", "w")
            f1.write(older.filecontents)
            f2.write(self.filecontents)
            f1.close()
            f2.close()
            (status, output) = commands.getstatusoutput("diff -u --label='%s' --label='%s' /tmp/a /tmp/b" % (ofname, nfname))
            data = output
        return data

    _diffdata = staticmethod(_diffdata)
    
    def to_filesys(self, destdir=None, use_dirs=1):
        """Writes the revision entry to the filesystem.

        This is one of the most important functions in the whole
        software. It tries to restore a given version to the
        filesystem, with almost all the attributes intact (ctime can't
        be restored, as far as I know).

        """

        if destdir is None:
            target = self.filename
        else:
            if use_dirs:
                target = forcejoin(destdir, self.filename)
            else:
                target = os.path.join(destdir, os.path.basename(self.filename))

        if self.filetype not in (stat.S_IFREG, stat.S_IFLNK, stat.S_IFCHR,
                                 stat.S_IFBLK, stat.S_IFIFO, stat.S_IFDIR):
            raise TypeError("Can't apply!")

        do_create  = True
        do_payload = True
        do_attrs   = True
        do_rename  = True
        
        if self.filetype == stat.S_IFDIR and os.path.exists(target):
            if not os.path.islink(target) and not os.path.isdir(target):
                print >>sys.stderr, "Can't overwrite non-directory with directory!"
                return
            else: # The directory already exists; we must restore ownership and attrs
                do_create  = False
                do_payload = False
                do_rename  = False
                newname = target
        else:
            if not os.path.islink(target) and os.path.isdir(target):
                print >>sys.stderr, "Can't overwrite directory with non-directory!"
                return
            newname = "%s.%07d" % (target, random.randint(0, 999999))
            retries = 0
            while os.path.exists(newname) and retries < 1000:
                newname = "%s.%07d" % (target, random.randint(0, 999999))
                retries += 1
            if os.path.exists(newname):
                print >>sys.stderr, "Can't create a temporary filename! Programmer error or race attack?"
                return

        oldumask = os.umask(0777)
        # try...finally for umask restoration
        try:
            must_remove_new = False
            # The ideea is the operation is done in five steps:
            # 1. creation of item; can fail; fatal
            # 2. if item is file, write contents; can fail; fatal
            # 3. change ownership; can fail; non-fatal
            # 4. change permissions and timestamps; shouldn't fail; fatal
            # 5. rename to target; can fail; fatal

            # Step 1
            if do_create:
                try:
                    if self.filetype == stat.S_IFIFO:
                        os.mkfifo(newname, 0)
                    elif self.filetype == stat.S_IFREG:
                        fd = os.open(newname, os.O_WRONLY|os.O_CREAT|os.O_EXCL|os.O_NOCTTY)
                    elif self.filetype == stat.S_IFLNK:
                        os.symlink(self.filecontents, newname)
                    elif self.filetype == stat.S_IFCHR or self.filetype == stat.S_IFBLK:
                        self._mknod(newname, self.mode, self.rdev)
                    elif self.filetype == stat.S_IFDIR:
                        os.mkdir(newname, 0)
                    else:
                        raise TypeError, "Don't know how to handle file type %s!" % self.filetype
                        return
                except OSError, e:
                    print >>sys.stderr, "Abort: error '%s' while creating temporary file." % e
                    return
                else:
                    must_remove_new = True
            # From now on, we must cleanup on exit (done via ...finally)
            # Step 2
            if do_payload:
                try:
                    if self.filetype == stat.S_IFREG:
                        os.write(fd, self.filecontents)
                        os.close(fd)
                except OSError, e:
                    print >>sys.stderr, "Abort: error '%s' while writing file contents." % e
                    return
            # Step 3
            if do_attrs:
                try:
                    os.lchown(newname, self.uid, self.gid)
                except OSError, e:
                    print >>sys.stderr, "Warning: error '%s' while modifying temporary file ownership." % e
                # Step 4
                try:
                    os.chmod(newname, self.mode)
                    os.utime(newname, (self.atime, self.mtime))
                except OSError, e:
                    print >>sys.stderr, "Abort: error '%s' while modifying temporary file attributes." % e
                    return
            # Step 5
            if do_rename:
                try:
                    os.rename(newname, target)
                except OSError, e:
                    print >>sys.stderr, "Abort: error '%s' while renaming" % e
                    raise
                else:
                    must_remove_new = False # We managed to finish!
        finally:
            os.umask(oldumask)
            if must_remove_new:
                try:
                    os.unlink(newname)
                except OSError, e:
                    print >>sys.stderr, "Error while cleaning-up: %s" % e
        return
        
    def diff(self, older):
        obuff = StringIO()
        if not isinstance(older, RevEntry):
            raise TypeError("Invalid diff!")
        if self == older:
            return ""
        if self.revno is None:
            newrev = 'current'
        else:
            newrev = "rev %s" % self.revno
        orev = "rev %s" % older.revno
        if self.filetype != older.filetype:
            return "File type has changed: from %s to %s" % (older.filetype, self.filetype)
        if self.filetype == stat.S_IFREG or self.filetype == stat.S_IFDIR:
            if self.filecontents != older.filecontents:
                obuff.write("== File contents diff: ==\n")
                data = self._diffdata(
                    older.filecontents, self.filecontents,
                    older.filename, self.filename,
                    "%s (%s)" % (time.ctime(older.mtime), orev),
                    "%s (%s)" % (time.ctime(self.mtime), newrev)
                    )
                obuff.write(data)
                obuff.write("\n")
        elif self.filetype == stat.S_IFLNK:
            if self.filecontents != older.filecontents:
                obuff.write("Symlink target changed from '%s' to '%s'" % (older.filecontents, self.filecontents))
        else:
            obuff.write("Don't know how to diff")
        obuff.write("== File metadata diff ==\n")
        for i in ('mtime', 'atime', 'uid', 'gid', 'mode'):
            oval = getattr(older, i)
            nval = getattr(self, i)
            if oval != nval:
                obuff.write("- %s: %s\n" % (i, oval))
                obuff.write("+ %s: %s\n" % (i, nval))
        return obuff.getvalue()

    def __eq__(self, other):
        if not isinstance(other, RevEntry):
            return NotImplemented
        # Do not test atime, it's irrelevant
        for i in ('filename', 'filetype', 'mode',
                  'mtime', 'uid', 'gid', 'rdev',
                  'filecontents'):
            if getattr(self, i) != getattr(other, i):
                return False
        return True

    def isdir(self):
        return self.filetype == stat.S_IFDIR

    def isreg(self):
        return self.filetype == stat.S_IFREG

    def islnk(self):
        return self.filetype == stat.S_IFLNK

    def isblk(self):
        return self.filetype == stat.S_ISBLK

    def ischr(self):
        return self.filetype == stat.S_ISCHR

    def ififo(self):
        return self.filetype == stat.S_IFIFO

    def ifsock(self):
        return self.filetype == stat.S_IFSOCK

    def mode2str(self):
        def mapbit(mode, bit, y):
            if mode & bit:
                return y
            else:
                return '-'
            
        modemap = {
            stat.S_IFDIR: 'd',
            stat.S_IFREG: '-',
            stat.S_IFLNK: 'l',
            stat.S_IFBLK: 'b',
            stat.S_IFCHR: 'c',
            stat.S_IFIFO: 'p',
            stat.S_IFSOCK: 's',
            }
        tchar = modemap.get(self.filetype, '?')
        tchar += mapbit(self.mode, stat.S_IRUSR, 'r')
        tchar += mapbit(self.mode, stat.S_IWUSR, 'w')
        if self.mode & stat.S_ISUID:
            tchar += 'S'
        else:
            tchar += mapbit(self.mode, stat.S_IXUSR, 'x')
        tchar += mapbit(self.mode, stat.S_IRGRP, 'r')
        tchar += mapbit(self.mode, stat.S_IWGRP, 'w')
        if self.mode & stat.S_ISGID:
            tchar += 'S'
        else:
            tchar += mapbit(self.mode, stat.S_IXGRP, 'x')
        tchar += mapbit(self.mode, stat.S_IROTH, 'r')
        tchar += mapbit(self.mode, stat.S_IWOTH, 'w')
        tchar += mapbit(self.mode, stat.S_IXOTH, 'x')
        return tchar
            
class AreaRev(object):
    __slots__ = ["area", "revno", "logmsg", "uid", "gid",
                 "commiter", "_ctime", "itemids"]

    def __init__(self, area=None, logmsg=None, commiter=None):
        if area is None and logmsg is None:
            # manual initialization
            return
        self.area = area.id
        self.revno = area.revno + 1
        self.logmsg = logmsg
        self._ctime = DateTime.utc()
        self.uid = os.getuid()
        self.gid = os.getgid()
        if commiter is None:
            try:
                self.commiter = os.getlogin()
            except OSError, e:
                self.commiter = '<unknown>'
        else:
            self.commiter = commiter
        self.itemids = []

    def _set_ctime(self, val):
        if isinstance(val, types.StringTypes):
            val = DateTime.ISO.ParseDateTimeUTC(val)
        self._ctime = val

    def _get_ctime(self):
        return self._ctime

    ctime = property(_get_ctime, _set_ctime, None,
                     "The creation time of this revision")
