//  Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>
#include <assert.h>
#include <memory>
#include <iostream>
#include <assert.h>
#include <glib.h>
#include <gtkmm/main.h>
#include <gtkmm.h>
#include <libsoupmm/wrap_init.h>

#include "main.h"
#include "timing.h"
#include "cbc-osd-connector.h"
#include "cbc-show-list.h"
#include "cbc-show.h"
#include "ucompose.hpp"
#include "tray-icon.h"

struct Main::Impl
{
    std::auto_ptr<Gtk::Main> gtk_main;
    TrayIcon *tray_icon;
    sigc::connection on_timer_registered(Timing::timer_slot s,
					 int msecs_interval);
    CBCOSDConnector *cbcosd;
    CBCShowList *shown_news;
    CBCShowList *shown_sports;
};


static Main *singleton;

Main::Main(int &argc, char **&argv)
    : impl(new Impl)
{
    version_appears_on_the_command_line = false;
    singleton = this;

    Glib::set_prgname("CBC Live Event Notifier");
    Glib::thread_init();
    try
      {
        Glib::OptionContext context("");
        Glib::OptionEntry entry;
        Glib::OptionGroup options (PACKAGE_NAME, "CBC-OSD Options", "Command-line options for CBC-OSD");
        entry.set_long_name ("version");
        entry.set_short_name ('V');
        entry.set_description ("Show version and exit.");
        options.add_entry (entry, version_appears_on_the_command_line);
        context.add_group(options);
        context.set_summary("This program notifies us when there is a live news event streaming on cbc.ca.");

        try
          {
            context.parse(argc, argv);
          }
        catch(const Glib::Error& ex)
          {
            std::cout << ex.what() << std::endl;
            std::cout << "Try `" PACKAGE_NAME 
              " --help' for more information." << std::endl;
            exit(0);
          }

        if (version_appears_on_the_command_line)
          {
            std::cout << PACKAGE_NAME << " " << PACKAGE_VERSION << std::endl;
            std::cout << 
              "This is free software: " <<
              "you are free to change and redistribute it." << std::endl;
            std::cout << 
              "There is NO WARRANTY, to the extent permitted by law." << 
              std::endl;
            exit(0);
          }

	Timing::instance().timer_registered.connect(
	    sigc::mem_fun(*impl, &Main::Impl::on_timer_registered));

	impl->gtk_main.reset(new Gtk::Main(argc, argv, context));
    Soup::wrap_init();

    }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }
}

Main::~Main()
{
    delete impl;
    singleton = 0;
}

Main &Main::instance()
{
    assert(singleton != 0);
    return *singleton;
}

bool Main::iterate_main_loop()
{
    try
    {
	impl->gtk_main->iteration(false);
    }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }

    return true;
}

void Main::on_connect_succeeded(bool success)
{
  if (!success)
    stop_main_loop();
  //std::cerr << "Got Id as " << impl->cbcosd->get_id() << std::endl;
  //std::cerr << "Got News Id as " << impl->cbcosd->get_news_id() << std::endl;
  //std::cerr << "Got Sports Id as " << impl->cbcosd->get_sports_id() << std::endl;
  impl->cbcosd->news_polled.connect(sigc::bind(sigc::mem_fun(*this, &Main::on_polled), impl->cbcosd->get_news_id(), impl->shown_news));
  impl->cbcosd->sports_polled.connect(sigc::bind(sigc::mem_fun(*this, &Main::on_polled), impl->cbcosd->get_sports_id(), impl->shown_sports));
  impl->cbcosd->start_polling(CBCOSDConnector::get_default_polling_period());
}

void Main::on_polled(const char *data, int datalen, Glib::ustring category_id, CBCShowList *shown)
{
  CBCShowList *shows = CBCShowList::create_from_json(data, datalen);
  for (CBCShowList::iterator i = shows->begin(); i != shows->end(); i++)
    {
      if ((*i).is_showing())
        {
          if (shown->contains((*i).get_id()) == false)
            {
              Glib::ustring url = 
                String::ucompose("%1#/News/%2/ID=%3", CBC_VIDEO_BASE,
                                 category_id, (*i).get_id());
              impl->cbcosd->notify((*i).get_title(), (*i).get_id(), url);
              shown->push_back(*i);
            }
        }
    }
  delete shows;
}

void Main::start_main_loop()
{
    try
      {
        impl->cbcosd = CBCOSDConnector::create(true, false);
        impl->shown_news = new CBCShowList();
        impl->shown_sports = new CBCShowList();
        impl->cbcosd->connect_succeeded.connect(sigc::mem_fun(*this, &Main::on_connect_succeeded));
        impl->cbcosd->connect();
        impl->tray_icon = new TrayIcon();
        impl->gtk_main->run();
      }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }
}

void Main::stop_main_loop()
{
    try
    {
	impl->gtk_main->quit();
        delete impl->cbcosd;
        delete impl->shown_news;
        delete impl->shown_sports;
        delete impl->tray_icon;
    }
    catch (const Glib::Error &ex) {
	std::cerr << ex.what() << std::endl;
    }
}

std::string Main::get_data_path()
{
  return CBCOSD_DATADIR;
}

sigc::connection Main::Impl::on_timer_registered(Timing::timer_slot s,
						 int msecs_interval)
{
    return Glib::signal_timeout().connect(s, msecs_interval);
}

bool Main::get_watching_news()
{
  return Main::instance().impl->cbcosd->get_watch_news();
}

bool Main::get_watching_sports()
{
  return Main::instance().impl->cbcosd->get_watch_sports();
}
    
void Main::set_sports_toggled()
{
  bool watch = impl->cbcosd->get_watch_sports();
  impl->cbcosd->set_watch_sports(!watch);
}

void Main::set_news_toggled()
{
  bool watch = impl->cbcosd->get_watch_news();
  impl->cbcosd->set_watch_news(!watch);
}

void Main::restart_polling()
{
  impl->cbcosd->stop_polling();
  delete impl->shown_news;
  impl->shown_news = new CBCShowList();
  delete impl->shown_sports;
  impl->shown_sports = new CBCShowList();
  impl->cbcosd->start_polling(CBCOSDConnector::get_default_polling_period());
}
