// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef STACK_H
#define STACK_H

#include <list>
#include <vector>
#include "vector.h"
#include <sigc++/trackable.h>
#include <sigc++/signal.h>
#include <sstream>

#include "UniquelyIdentified.h"
#include "Ownable.h"
#include "Movable.h"

class Player;
class Path;
class Army;
class XML_Helper;
class Temple;
class Hero;
class Item;

//! A set of up to eight Army units that move as a single entity on the map.
/** 
 * While Army units are the actual troops you command, they always belong to a
 * stack. The stack holds these armies together in one object. The player
 * usually doesn't command the armies but the stack, so all functionality and
 * data which affects player's controls is bundled in the stack class. Among
 * this is the location of the units, the intended movement path, and more.
 */

class Stack : public ::UniquelyIdentified, public Movable, public Ownable, public std::list<Army*>, public sigc::trackable
{
    public:
	//! The xml tag of this object in a saved-game file.
	static Glib::ustring d_tag; 

        /** 
	 * Make a new stack.
         * 
         * @param player       A pointer to the owning player.
         * @param pos          The position on the map where the stack is to 
	 *                     be created.
         */
	//! Default constructor.
        Stack(Player* player, Vector<int> pos);

	/**
	 * Copy the whole stack into a new stack.  This method performs a 
	 * deep copy of the stack's Army units.
	 */
        //! Copy constructor.
        Stack(const Stack& s);

        //! Loading constructor.
        Stack(XML_Helper* helper);

	//! Destructor.
        ~Stack();

	// Get Methods

        //! How many army units can be put into this stack?
        guint32 getMaxArmiesToJoin() const;

        bool hasDeadArmies() const;

	// Set Methods

        //! Change the loyalty of the stack.
        void setPlayer(Player* p);

	// Methods that operate on class data and modify the class
	
        //! Erase the stack, deleting the Army units too.
        void flClear();

        void moveToDest(Vector<int> dest);
        /** 
	 * Erase an Army unit from the Stack, and free the contents of 
	 * the Army unit too (e.g. Items a Hero might be carrying).
	 *
	 * @param it   The place in the Stack to erase.
	 *
	 * @return The place in the stack that was erased.
         */
	//! Erase an Army unit from the list.
        iterator flErase(iterator object);

	//! Add an army to this stack.
	/**
	 * This method should be used instead of push_back.
	 */
	void add(Army *army);

	// Methods that operate on class and do not modify the class

        //! Does the stack contain this kind of army?
        bool hasArmyType(guint32 army_type) const;

        //! Save the stack to an opened saved-game file.
        bool save(XML_Helper* helper) const;

	//! Go find the army with this identifier in the stack and return it.
        Army* getArmyById(guint32 id) const;
        
	//! Does everything in this stack look okay?
	bool validate() const;

	//! Does this stack have 8 units in it?
	bool isFull() const;

        bool hasGeneral() const;

        double getMaximumMoves() const;
        double getMoveBonus() const;
        double getAttackBonus() const;
        int getMoveRadius(int die_value) const;
        double getPowerBonus() const;
        bool canAttackAcrossWater() const;
        double getAttackBonusWaterPenalty() const;
        bool canMoveAcrossWater() const;
        bool canAttackOverEnemies() const;
        bool canDestroyBrick() const;
        guint32 getMinimumDieToBreakBrick() const;

	// Signals

	//! Emitted when this stack dies.
        sigc::signal<void, Stack*> sdying;

	//! Emitted when this stack is about to move one step
	sigc::signal<void, Stack*> smoving;

	//! Emitted when this stack has finished moving that one step
	sigc::signal<void, Stack*> smoved;


	// Static Methods
        

	//! Create a stack with an id that isn't unique.
	static Stack* createNonUniqueStack(Player *player, Vector<int> pos);

    private:    

	//! Private constructor.
	Stack(guint32 id, Player* player, Vector<int> pos);

        //! Callback for loading the object from an opened saved-game file.
        bool load(Glib::ustring tag, XML_Helper* helper);
    
};

#endif // STACK_H

// End of file
