//  Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef GUI_GAME_WINDOW_H
#define GUI_GAME_WINDOW_H

#include <memory>
#include <vector>
#include <sigc++/signal.h>
#include <sigc++/trackable.h>
#include <sigc++/connection.h>
#include <gtkmm.h>

#include "game-parameters.h"
#include "stack.h"
#include "vector.h"
#include "army.h"
#include "GraphicsCache.h"
#include "GameScenario.h"

class Game;
class Fight;
class Player;
class NextTurn;
class DiceBox;

class GameWindow
{
 public:
    GameWindow();
    ~GameWindow();

    void show();
    void hide();

    // initialize the big map widget 
    void init(int width, int height);

    // setup a new game
    void new_game(GameScenario *game_scenario, NextTurn *next_turn);

    // load the game
    void load_game(GameScenario *game_scenario, NextTurn *next_turn);

    // emitted when the game has ended and it is time to show the splash again
    sigc::signal<void> game_ended;

    // emitted when the game has ended and we want to start a new game.
    sigc::signal<void> game_ended_start_new;
    
    sigc::signal<void> quit_requested;

    sigc::signal<void> sdl_initialized;
    
    Gtk::Window *get_window() const {return window;};

    void redraw();

 private:
    Gtk::Window* window;
    Gtk::ImageMenuItem *new_game_menuitem;
    Gtk::ImageMenuItem *load_game_menuitem;
    Gtk::ImageMenuItem *save_game_menuitem;
    Gtk::ImageMenuItem *save_as_game_menuitem;
    Gtk::ImageMenuItem *quit_menuitem;
    Gtk::ImageMenuItem *preferences_menuitem;
    Gtk::ImageMenuItem *online_help_menuitem;
    Gtk::ImageMenuItem *about_menuitem;
    Gtk::Image *map_image;
    Gtk::EventBox *map_eventbox;
    Gtk::Label *p1_label;
    Gtk::Label *p2_label;
    Gtk::Box *boxp1;
    Gtk::Box *boxp2;
    DiceBox *p1_dicebox;
    DiceBox *p2_dicebox;
    Gtk::Statusbar *statusbar;

    Glib::ustring current_save_filename;

    Game* game;
    std::list<sigc::connection> connections;
    Player *game_winner;
    unsigned int selected_die_value;
    Rectangle selected_move_box;
    Rectangle selected_attack_box;
    int die1_value;
    int die2_value;
    int die3_value;
    int die4_value;
    
    bool sdl_inited;
    Vector<int> current_tile;

    bool on_delete_event(GdkEventAny *e);
    
    void on_load_game_activated();
    void on_save_game_activated();
    void on_save_game_as_activated();
    void on_quit_activated();
    void on_new_game_activated();
    void on_game_stopped();
    void on_preferences_activated();
    void on_help_about_activated();
    void on_online_help_activated();

    void on_message_requested(Glib::ustring msg);
    
    // game callbacks
    void on_next_player_turn(Player *player, unsigned int turn_number);
    void on_game_loaded(Player *player);
    void on_game_over(Player *winner);
    void on_player_died(Player *player);
    void on_fight_started(Fight::Result result, int defender_roll, double attacker_power, double defender_power);
    void on_stack_moves(Stack *s, std::list<Vector<int> > points, Vector<int> blocked, Vector<int> stopped_short);
    void on_raft_moves(Stack *s, std::list<Vector<int> > points);

    //dicebox callbacks
    void on_die_selected_for_move(int die_value);
    void on_die_selected_for_attack(int die_value);
    void on_die_selected(int die_value);
    void on_die_used();
    void on_no_dice_left();

    //map callbacks
    bool on_map_mouse_button_event(GdkEventButton *e);
    bool on_map_mouse_motion_event(GdkEventMotion *event);
    
    //helpers
    bool setup_game(GameScenario *game_scenario, NextTurn *nextTurn);
    void setup_signals(GameScenario *game_scenario);
    void stop_game(Glib::ustring action);
    void setup_menuitem(Gtk::MenuItem*, sigc::slot<void> , sigc::signal<void, bool> &);

    void give_some_cheese(Player *game_winner);

    DiceBox *get_active_dicebox();
    DiceBox *get_inactive_dicebox();
    Stack *stack_is_currently_moving;
    Raft* raft_is_currently_moving;

public:
    Glib::ustring stop_action; //hackhackhack
    Glib::ustring d_scenario;
    Glib::ustring d_player_name;
    Glib::ustring d_load_filename;
};

#endif
