//  Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <assert.h>
#include <sigc++/functors/mem_fun.h>
#include <sigc++/bind.h>
#include <errno.h>
#include <string.h>

#include "driver.h"

#include <iostream>
#include "main.h"
#include "game-window.h"
#include "defs.h"
#include "GraphicsCache.h"
#include "GameScenario.h"
#include "counter.h"
#include "File.h"
#include "playerlist.h"
#include "xmlhelper.h"
#include "Configuration.h"

#include "NextTurnHotseat.h"
#include "game.h"
#include "timed-message-dialog.h"

Driver::Driver (Glib::ustring load_filename)
{
  game_window = NULL;

  d_load_filename = load_filename;

  run ();
}

void
Driver::get_default (int num_players, GameParameters & g)
{
  GameParameters::Player p;
  if (num_players <= 1 || num_players > (int) MAX_PLAYERS)
    num_players = (int) MAX_PLAYERS;
  for (unsigned int i = 0; i < (unsigned int) num_players; i++)
    {
      p.type = GameParameters::Player::HUMAN;
      p.id = i;
      switch (p.id)
	{
	case 0:
	  p.name = "Player One";
	  break;
	case 1:
	  p.name = "Player Two";
	  break;
	}
      g.players.push_back (p);
    }
  g.map_path = File::getMapFile ("default.map");
}

void
Driver::run ()
{
  if (d_load_filename.empty () == false)
    {
      GameParameters g;
      get_default (2, g);
      g.map_path = d_load_filename;
      size_t found = d_load_filename.find (".map");
      if (found != Glib::ustring::npos)
	on_new_game_requested (g);
      else
	{
	  found = d_load_filename.find (SAVE_EXT);
	  if (found != Glib::ustring::npos)
	    on_load_requested (d_load_filename);
	  else
	    on_new_game_requested (g);
	}
    }
  else
    {
      GameParameters g;
      get_default (2, g);
      on_new_game_requested (g);
    }
  return;
}

Driver::~Driver ()
{
  if (game_window)
    {
      delete game_window;
      game_window = NULL;
    }
}

GameScenario *
Driver::create_new_scenario (GameParameters & g, GameScenario::PlayMode m)
{
  bool update_uuid = false;
  if (g.map_path.empty ())
    return NULL;

  bool broken = false;

  GameScenario *game_scenario = new GameScenario (g.map_path, broken);
  if (broken)
    return NULL;

  game_scenario->setName (g.name);
  game_scenario->setPlayMode (m);

  if (game_scenario->getRound () == 0)
    {
      if (update_uuid)
	game_scenario->setNewRandomId ();
      game_scenario->initialize (g);
    }
  return game_scenario;
}

void
Driver::on_new_game_requested (GameParameters g)
{
  init_game_window ();
  game_window->show ();
  GameScenario *game_scenario =
    create_new_scenario (g, GameScenario::HOTSEAT);

  if (game_scenario == NULL)
    {
      TimedMessageDialog dialog (*game_window->get_window (),
				 "Corrupted saved game file.", 0);
      dialog.run ();
      dialog.hide ();
      game_window->hide ();
      return;
    }

  std::list < Glib::ustring > e, w;
  if (g.map_path != "" && game_scenario->validate (e, w) == false)
    {
      TimedMessageDialog dialog
	(*game_window->get_window (),
	 "Invalid map file.\n" "Please validate it manually.", 0);
      std::list < Glib::ustring >::iterator it = e.begin ();
      for (; it != e.end (); it++)
	{
	  printf ("error: %s\n", (*it).c_str ());
	}
      dialog.run ();
      dialog.hide ();
      game_window->hide ();
      return;
    }

  NextTurn *next_turn = new NextTurnHotseat ();

  game_window->new_game (game_scenario, next_turn);
}

void
Driver::on_load_requested (Glib::ustring filename)
{
  init_game_window ();
  game_window->show ();
  GameScenario *game_scenario = load_game (filename);
  if (game_scenario == NULL)
    {
      game_window->hide ();
      return;
    }


  if (game_scenario->getPlayMode () == GameScenario::HOTSEAT)
    game_window->load_game (game_scenario, new NextTurnHotseat ());
}

void
Driver::on_quit_requested ()
{
  if (game_window)
    game_window->hide ();

  exit(0);
}

void
Driver::on_game_ended ()
{
  GraphicsCache::deleteInstance ();
  on_quit_requested();
}

void
Driver::on_game_ended_and_start_new ()
{
  GraphicsCache::deleteInstance ();

  GameParameters g;
  get_default (2, g);
  g.map_path = File::getMapFile ("default.map");
  on_new_game_requested (g);
}

void
Driver::init_game_window ()
{
  if (game_window)
    delete game_window;
  game_window = new GameWindow;

  game_window->game_ended.
    connect (sigc::mem_fun (*this, &Driver::on_game_ended));
  game_window->game_ended_start_new.
    connect (sigc::mem_fun (*this, &Driver::on_game_ended_and_start_new));
  game_window->quit_requested.
    connect (sigc::mem_fun (*this, &Driver::on_quit_requested));

  //make the width+height suitable for the screen size.
  game_window->init (640, 480);

}

GameScenario *
Driver::load_game (Glib::ustring file_path)
{
  bool broken = false;
  GameScenario *game_scenario = new GameScenario (file_path, broken);

  if (broken)
    {
      TimedMessageDialog dialog (*game_window->get_window (),
				 "Corrupted saved game file.", 0);
      dialog.run ();
      dialog.hide ();
      return NULL;
    }
  return game_scenario;
}
