//  Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <gtkmm.h>

#include "dicebox.h"
#include "glade-helpers.h"
#include "GraphicsCache.h"

DiceBox::DiceBox()
{
  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(get_glade_path()
                                     + "/dicebox-window.ui");

  xml->get_widget("contents", contents);
  xml->get_widget("move_button", move_button);
  move_button->signal_toggled().connect
    (sigc::mem_fun(*this, &DiceBox::on_move_toggled));
  xml->get_widget("attack_button", attack_button);
  attack_button->signal_toggled().connect
    (sigc::mem_fun(*this, &DiceBox::on_attack_toggled));
  xml->get_widget("die1_button", die1_button);
  die1_button->signal_clicked().connect
    (sigc::bind(sigc::mem_fun(*this, &DiceBox::on_die_selected), 
                die1_button));
  xml->get_widget("die2_button", die2_button);
  die2_button->signal_clicked().connect
    (sigc::bind(sigc::mem_fun(*this, &DiceBox::on_die_selected), 
                die2_button));
  xml->get_widget("die3_button", die3_button);
  die3_button->signal_clicked().connect
    (sigc::bind(sigc::mem_fun(*this, &DiceBox::on_die_selected), 
                die3_button));
  xml->get_widget("die4_button", die4_button);
  die4_button->signal_clicked().connect
    (sigc::bind(sigc::mem_fun(*this, &DiceBox::on_die_selected), 
                die4_button));
  xml->get_widget("die1_image", die1_image);
  xml->get_widget("die2_image", die2_image);
  xml->get_widget("die3_image", die3_image);
  xml->get_widget("die4_image", die4_image);
  xml->get_widget("selected_die_button", selected_die_button);
  xml->get_widget("selected_die_image", selected_die_image);
  GraphicsCache *gc = GraphicsCache::getInstance();
  die1_image->property_pixbuf() = gc->getDicePic(0)->to_pixbuf();
  die2_image->property_pixbuf() = gc->getDicePic(0)->to_pixbuf();
  die3_image->property_pixbuf() = gc->getDicePic(0)->to_pixbuf();
  die4_image->property_pixbuf() = gc->getDicePic(0)->to_pixbuf();
  selected_die_image->property_pixbuf() = gc->getDicePic(0)->to_pixbuf();
  attack_button->set_sensitive(false);
  move_button->set_sensitive(false);
}

DiceBox::~DiceBox()
{
}

void DiceBox::on_move_toggled()
{
  if (move_button->get_active())
    {
      if (attack_button->get_active())
        attack_button->set_active(false);
      die_selected_for_move.emit(selected_die);
    }
}

void DiceBox::on_attack_toggled()
{
  if (attack_button->get_active())
    {
      if (move_button->get_active())
        move_button->set_active(false);
      die_selected_for_attack.emit(selected_die);
    }
}

void DiceBox::roll_defender_die(int land_on)
{
  set_sensitive(true);
  die1_button->set_sensitive(false);
  die2_button->set_sensitive(false);
  die3_button->set_sensitive(false);
  die4_button->set_sensitive(false);
  move_button->set_sensitive(false);
  attack_button->set_sensitive(false);
  shuffle_die(0, land_on);
}

void DiceBox::clear_selected_die()
{
  selected_die_image->property_pixbuf() = 
    GraphicsCache::getInstance()->getDicePic(0)->to_pixbuf();
  attack_button->set_active(false);
  attack_button->set_sensitive(false);
  move_button->set_active(false);
  move_button->set_sensitive(false);
  die1_button->set_sensitive(!die1_selected);
  die2_button->set_sensitive(!die2_selected);
  die3_button->set_sensitive(!die3_selected);
  die4_button->set_sensitive(!die4_selected);
  if (no_more_dice_to_select())
    no_dice.emit();
  else
    more_dice.emit();
}

void DiceBox::on_die_selected(Gtk::Button *die_button)
{
  die_button->set_sensitive(false);
  Gtk::Image *image = dynamic_cast<Gtk::Image*>(die_button->get_child());
  Glib::RefPtr<Gdk::Pixbuf> pixbuf = image->property_pixbuf();
  selected_die_image->property_pixbuf() = pixbuf;
  if (die_button == die1_button)
    {
    selected_die = 1;
    die1_selected = true;
    }
  else if (die_button == die2_button)
    {
    selected_die = 2;
    die2_selected = true;
    }
  else if (die_button == die3_button)
    {
    selected_die = 3;
    die3_selected = true;
    }
  else if (die_button == die4_button)
    {
    selected_die = 4;
    die4_selected = true;
    }

  image->property_pixbuf() = 
    GraphicsCache::getInstance()->getDicePic(0)->to_pixbuf();
  attack_button->set_sensitive(true);
  attack_button->set_active(false);
  move_button->set_sensitive(true);
  die1_button->set_sensitive(false);
  die2_button->set_sensitive(false);
  die3_button->set_sensitive(false);
  die4_button->set_sensitive(false);
  die_selected.emit(selected_die);
  move_button->set_active(true);
}

void DiceBox::shuffle_dice(int val1, int val2, int val3, int val4)
{
  selected_die_image->property_pixbuf() = 
    GraphicsCache::getInstance()->getDicePic(0)->to_pixbuf();
  selected_die = 0;
  shuffle_die (1, val1);
  shuffle_die (2, val2);
  shuffle_die (3, val3);
  shuffle_die (4, val4);
}

void DiceBox::shuffle_die(int die, int land_on)
{
  Gtk::Button *button;
  if (die == 1)
    {
    button = die1_button;
    die1_selected = false;
    }
  else if (die == 2)
    {
    button = die2_button;
    die2_selected = false;
    }
  else if (die == 3)
    {
    button = die3_button;
    die3_selected = false;
    }
  else if (die == 4)
    {
    button = die4_button;
    die4_selected = false;
    }
  else if (die == 0)
    button = selected_die_button;
  else
    return;

  if (selected_die == 0)
    button->set_sensitive(true);

  Gtk::Image *image = dynamic_cast<Gtk::Image*>(button->get_child());
  int max = (rand() % 17)  + 3;
  for (int i = 0; i < max; i++)
    {
      PixMask *die = GraphicsCache::getInstance()->getDicePic((rand() % 6) + 1);
      image->property_pixbuf() = die->to_pixbuf();
      image->show();
      while (g_main_context_iteration(NULL, FALSE)); //doEvents
      Glib::usleep(70000);
    }
  image->property_pixbuf() = 
    GraphicsCache::getInstance()->getDicePic(land_on)->to_pixbuf();
  image->show();
}
    
DiceBox::Mode DiceBox::get_mode()
{
  if (move_button->get_active())
    return DiceBox::MOVE;
  if (attack_button->get_active())
    return DiceBox::ATTACK;
  return DiceBox::NONE;

}

void DiceBox::clear_all_dice()
{
  set_sensitive(true);
  attack_button->set_active(false);
  move_button->set_active(false);
  set_sensitive(false);
  die1_selected = true;
  die2_selected = true;
  die3_selected = true;
  die4_selected = true;
  GraphicsCache *gc = GraphicsCache::getInstance();
  die1_image->property_pixbuf() = gc->getDicePic(0)->to_pixbuf();
  die2_image->property_pixbuf() = gc->getDicePic(0)->to_pixbuf();
  die3_image->property_pixbuf() = gc->getDicePic(0)->to_pixbuf();
  die4_image->property_pixbuf() = gc->getDicePic(0)->to_pixbuf();
  selected_die_image->property_pixbuf() = gc->getDicePic(0)->to_pixbuf();
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
}

void DiceBox::load(std::vector<guint32> dice, int selected_die_value)
{
  GraphicsCache *gc = GraphicsCache::getInstance();
  die1_selected = (dice[0] == 0);
  die2_selected = (dice[1] == 0);
  die3_selected = (dice[2] == 0);
  die4_selected = (dice[3] == 0);
  die1_button->set_sensitive(true);
  die1_image->property_pixbuf() = gc->getDicePic(dice[0])->to_pixbuf();
  die1_button->set_sensitive(!die1_selected);
  die2_button->set_sensitive(true);
  die2_image->property_pixbuf() = gc->getDicePic(dice[1])->to_pixbuf();
  die2_button->set_sensitive(!die2_selected);
  die3_button->set_sensitive(true);
  die3_image->property_pixbuf() = gc->getDicePic(dice[2])->to_pixbuf();
  die3_button->set_sensitive(!die3_selected);
  die4_button->set_sensitive(true);
  die4_image->property_pixbuf() = gc->getDicePic(dice[3])->to_pixbuf();
  die4_button->set_sensitive(!die4_selected);
  selected_die_image->property_pixbuf() = gc->getDicePic(selected_die_value)->to_pixbuf();
  if (selected_die_value > 0)
    {
      die1_button->set_sensitive(false);
      die2_button->set_sensitive(false);
      die3_button->set_sensitive(false);
      die4_button->set_sensitive(false);
      move_button->set_sensitive(true);
      attack_button->set_sensitive(true);
    }
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
}
