//  Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <stdlib.h>
#include <fstream>
#include <iostream>
#include <sys/stat.h>
#include <sigc++/functors/mem_fun.h>

#include "Configuration.h"

#include "xmlhelper.h"
#include "defs.h"
#include "File.h"

Glib::ustring Configuration::d_tag = "armyrc";
Glib::ustring Configuration::configuration_file_path =
Glib::ustring (getenv ("HOME")) + "/.armyrc";
Glib::ustring Configuration::s_dataPath = ARMY_DATADIR;
Glib::ustring Configuration::s_savePath =
Glib::ustring (getenv ("HOME")) + Glib::ustring ("/.army/");
bool Configuration::s_musicenable = true;
Glib::ustring Configuration::s_filename = "";

Configuration::Configuration ()
{
}

Configuration::~Configuration ()
{
}

// check if file exists and parse it

bool
Configuration::loadConfigurationFile (Glib::ustring fileName)
{
  s_filename = fileName;

  std::ifstream in (fileName.c_str ());
  if (in)
    {
      //parse the file
      XML_Helper helper (fileName.c_str (), std::ios::in);
      helper.registerTag (d_tag,
			  sigc::mem_fun (*this,
					 &Configuration::parseConfiguration));

      return helper.parseXML ();
    }
  else
    return false;
}

bool
Configuration::saveConfigurationFile (Glib::ustring filename)
{
  bool retval = true;

  XML_Helper helper (filename, std::ios::out);

  //start writing
  retval &= helper.begin (ARMY_CONFIG_VERSION);
  retval &= helper.openTag (d_tag);

  //save the values 
  retval &= helper.saveData ("datapath", s_dataPath);
  retval &= helper.saveData ("savepath", s_savePath);
  retval &= helper.saveData ("musicenable", s_musicenable);
  retval &= helper.closeTag ();

  if (!retval)
    {
      std::cerr << "Configuration: Something went wrong while saving.\n";
      return false;
    }

  helper.close ();

  return true;
}

bool
Configuration::parseConfiguration (Glib::ustring tag, XML_Helper * helper)
{
  Glib::ustring temp;
  bool retval;

  if (helper->getVersion () != ARMY_CONFIG_VERSION)
    {
      std::cerr << "Configuration file has wrong version, we want ";
      std::cerr << ARMY_SAVEGAME_VERSION << ",\n";
      std::cerr << "Configuration file offers " << helper->
	getVersion () << ".\n";

      Glib::ustring orig = s_filename;
      Glib::ustring dest = s_filename + ".OLD";

      std::ofstream ofs (dest.c_str ());
      std::ifstream ifs (orig.c_str ());
      ofs << ifs.rdbuf ();
      ofs.close ();

      return false;
    }

  //get the paths
  retval = helper->getData (temp, "datapath");
  if (retval)
    s_dataPath = temp;

  retval = helper->getData (temp, "savepath");
  if (retval)
    s_savePath = temp;


  // parse musicsettings
  retval = helper->getData (s_musicenable, "musicenable");
  return retval;
}

void
initialize_configuration ()
{
  Configuration conf;

  bool foundconf =
    conf.loadConfigurationFile (Configuration::configuration_file_path);
  if (!foundconf)
    {
      bool saveconf =
	conf.saveConfigurationFile (Configuration::configuration_file_path);
      if (!saveconf)
	{
	  std::cerr << "Couldn't save the new configuration file..." << std::
	    endl;
	  std::
	    cerr << "Check permissions of your home directory....aborting!" <<
	    std::endl;
	  exit (-1);
	}
      else
	std::
	  cerr << "Created the standard configuration file " <<
	  Configuration::configuration_file_path << std::endl;
    }

  //Check if the save game directory exists. If not, try to create it.

  if (File::create_dir (Configuration::s_savePath) == false)
    {
      std::cerr << "Couldn't create save game directory ";
      std::cerr << Configuration::s_savePath << ".\n";
      std::
	cerr << "Check permissions and the entries in your armyrc file!"
	<< std::endl;
      exit (-1);
    }
  //Check if the personal armyset directory exists. If not, try to create it.
  if (File::create_dir (File::getUserArmysetDir ()) == false)
    {
      std::cerr << "Couldn't create personal armyset directory ";
      std::cerr << File::getUserArmysetDir () << ".\n";
      std::
	cerr << "Check permissions and the entries in your armyrc file!"
	<< std::endl;
      exit (-1);
    }

  //Check if the personal maps directory exists. If not, try to create it.
  if (File::create_dir (File::getUserMapDir ()) == false)
    {
      std::cerr << "Couldn't create personal map directory ";
      std::cerr << File::getUserMapDir () << ".\n";
      std::
	cerr << "Check permissions and the entries in your armyrc file!"
	<< std::endl;
      exit (-1);
    }

}
