/*
 *   This file is part of AkariXB
 *   Copyright 2015-2018  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "mainwindow.h"


MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{
    this->setWindowTitle("AkariXB");
    this->setWindowIcon(QIcon(":/icon/64x64/akarixb.png"));
    this->setMinimumSize(500, 400);

    // Ensure closing a dialog while the main window is hidden, won't end the program
    qApp->setQuitOnLastWindowClosed(false);

    m_globalObject = new GlobalObject(this);
    connect(m_globalObject, &GlobalObject::showInStatusBar,
            this, &MainWindow::setStatusBarMessage);


#if 0 // Set to 1 to test the fallback icons
    QIcon::setThemeSearchPaths(QStringList() << "./");
#endif

    setupXmppClient();

    // It's important to initialize this before creating the handlers, which use it
    this->variableParser = new VariableParser(this);
    m_globalObject->setVariableParser(variableParser);

    this->messageHandler = new MessageHandler(m_globalObject, this);
    this->commandHandler = new CommandHandler(m_globalObject, this);
    this->activityHandler = new ActivityHandler(m_globalObject, this);


    // Page 8, runtime log, needed before others for early logging
    this->logModule = new LogModule(this);
    logModule->addToLog(QString("Qt v%1 (%2), QXMPP v%3, %4.")
                        .arg(qVersion(),
                             qApp->platformName(),
                             QXmppVersion(),
                             QSysInfo::prettyProductName()));
    connect(m_globalObject, &GlobalObject::logMessage,
            logModule, &LogModule::addToLog);

    createDataDir();


    this->configDialog = new ConfigDialog(m_globalObject, this);



    // Page 1, general
    this->generalModule = new GeneralModule(m_globalObject, this);

    // Page 2, MUCs
    this->roomModule = new RoomModule(m_globalObject, this);

    // Page 3, private chats
    this->chatModule = new ChatModule(m_globalObject, this);
    connect(chatModule, SIGNAL(unreadCountChanged(int)),
            this, SLOT(setChatsTitle(int)));

    // Page 4, commands
    this->commandModule = new CommandModule(this->commandHandler,
                                            m_globalObject,
                                            this);

    // Page 5, conversations
    this->conversationModule = new ConversationModule(this);

    // Page 6, activities
    this->activityModule = new ActivityModule(this->activityHandler,
                                              m_globalObject,
                                              this);

    // Page 7, identity
    this->identityModule = new IdentityModule(m_globalObject, this);



    // Categories panel
    this->categoriesList = new QListWidget(this);
    categoriesList->setViewMode(QListView::ListMode);
    categoriesList->setFlow(QListView::TopToBottom);
    categoriesList->setWrapping(false);
    categoriesList->setIconSize(QSize(48, 48));
    categoriesList->setMinimumWidth(120); // TMP/FIXME
    categoriesList->setWordWrap(true);
    categoriesList->setMovement(QListView::Static);
    categoriesList->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    categoriesList->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    categoriesList->addItem(tr("General"));
    categoriesList->item(0)->setIcon(QIcon::fromTheme("user-home",
                                                      QIcon(":/images/section-general.png")));
    categoriesList->addItem(tr("Rooms"));
    categoriesList->item(1)->setIcon(QIcon::fromTheme("system-users",
                                                      QIcon(":/images/section-rooms.png")));
    categoriesList->addItem(QString()); // Chats, title set later
    categoriesList->item(2)->setIcon(QIcon::fromTheme("system-users",
                                                      QIcon(":/images/section-rooms.png")));
    this->setChatsTitle(0);
    categoriesList->addItem(tr("Commands"));
    categoriesList->item(3)->setIcon(QIcon::fromTheme("system-run",
                                                      QIcon(":/images/section-general.png")));
    categoriesList->addItem(tr("Conversation"));
    categoriesList->item(4)->setIcon(QIcon::fromTheme("view-conversation-balloon",
                                                      QIcon(":/images/section-rooms.png")));
    categoriesList->addItem(tr("Activities"));
    categoriesList->item(5)->setIcon(QIcon::fromTheme("view-categories",
                                                      QIcon(":/images/section-general.png")));
    categoriesList->addItem(tr("Identity"));
    categoriesList->item(6)->setIcon(QIcon::fromTheme("user-identity",
                                                      QIcon(":/images/no-avatar.png")));
    categoriesList->addItem(tr("Log"));
    categoriesList->item(7)->setIcon(QIcon::fromTheme("text-x-log",
                                                      QIcon(":/images/section-log.png")));

    // Pages on right side
    this->stackedWidget = new QStackedWidget(this);
    stackedWidget->addWidget(generalModule);
    stackedWidget->addWidget(roomModule);
    stackedWidget->addWidget(chatModule);
    stackedWidget->addWidget(commandModule);
    stackedWidget->addWidget(conversationModule);
    stackedWidget->addWidget(activityModule);
    stackedWidget->addWidget(identityModule);
    stackedWidget->addWidget(logModule);
    connect(categoriesList, SIGNAL(currentRowChanged(int)),
            stackedWidget, SLOT(setCurrentIndex(int)));
    connect(categoriesList, SIGNAL(currentRowChanged(int)),
            this, SLOT(setTitleAndTrayInfo()));

    this->mainLayout = new QHBoxLayout();
    mainLayout->addWidget(categoriesList, 1);
    mainLayout->addWidget(stackedWidget,  5);

    this->mainWidget = new QWidget(this);
    mainWidget->setLayout(mainLayout);
    this->setCentralWidget(mainWidget);


    loadSettings();


    this->connectionDialog = new ConnectionDialog(m_userJid,
                                                  m_userPassword,
                                                  m_userAutoconnect,
                                                  m_userResource,
                                                  m_userPriority,
                                                  this);
    connect(connectionDialog, SIGNAL(connectionRequested(QString,QString,bool,QString,int)),
            this, SLOT(connectToServer(QString,QString,bool,QString,int)));
    connect(generalModule, SIGNAL(connectionRequested()),
            connectionDialog, SLOT(show()));
    connect(generalModule, &GeneralModule::disconnectionRequested,
            this, &MainWindow::disconnectFromServer);


    createMenus();
    createTrayIcon();
    createStatusBarWidgets();


    this->categoriesList->setCurrentRow(0);
    this->setTitleAndTrayInfo();

    /*
     * If initial connection hangs for some reason, break it and retry after
     * a relatively short time, to avoid connection limbo.
     *
     * See https://github.com/qxmpp-project/qxmpp/issues/166
     */
    m_initialConnectionTimer = new QTimer(this);
    m_initialConnectionTimer->setSingleShot(true);
    m_initialConnectionTimer->setInterval(60000); // 60 sec, hardcoded, FIXME
    connect(m_initialConnectionTimer, &QTimer::timeout,
            this, &MainWindow::onConnectionTimerAbort);


    this->statusBar()->showMessage(tr("Ready."));


    if (m_userJid.isEmpty())
    {
        if (!m_globalObject->getHideInTray())
        {
            connectionDialog->show();
        }
    }
    else if (m_userAutoconnect)
    {
        this->autoConnectToServer();
    }

    qDebug() << "MainWindow created";
}


MainWindow::~MainWindow()
{
    this->saveSettings();

    qDebug() << "MainWindow destroyed";
}


void MainWindow::createMenus()
{
    // Session
    this->sessionConnectAction = new QAction(QIcon::fromTheme("network-connect",
                                                              QIcon(":/images/button-online.png")),
                                             tr("Connect"),
                                             this);
    sessionConnectAction->setShortcut(QKeySequence("Ctrl+Shift+C"));
    connect(sessionConnectAction, SIGNAL(triggered()),
            connectionDialog, SLOT(show()));

    this->sessionDisconnectAction = new QAction(QIcon::fromTheme("network-disconnect",
                                                                 QIcon(":/images/button-offline.png")),
                                                tr("Disconnect"),
                                                this);
    sessionDisconnectAction->setDisabled(true);
    sessionDisconnectAction->setShortcut(QKeySequence("Ctrl+Shift+D"));
    connect(sessionDisconnectAction, &QAction::triggered,
            this, &MainWindow::disconnectFromServer);


    this->sessionQuitAction = new QAction(QIcon::fromTheme("application-exit"),
                                   tr("Quit"),
                                   this);
    sessionQuitAction->setShortcut(QKeySequence("Ctrl+Q"));
    connect(sessionQuitAction, SIGNAL(triggered()),
            this, SLOT(quitProgram()));


    this->sessionMenu = new QMenu(tr("&Session"), this);
    sessionMenu->addAction(sessionConnectAction);
    sessionMenu->addAction(sessionDisconnectAction);
    sessionMenu->addSeparator();
    sessionMenu->addAction(sessionQuitAction);
    this->menuBar()->addMenu(sessionMenu);


    // Settings
    this->settingsConfigAction = new QAction(QIcon::fromTheme("configure",
                                                              QIcon(":/images/button-configure.png")),
                                             tr("Configure AkariXB"),
                                             this);
    settingsConfigAction->setShortcut(QKeySequence("Ctrl+Shift+S"));
    connect(settingsConfigAction, SIGNAL(triggered()),
            configDialog, SLOT(show()));

    this->settingsMenu = new QMenu(tr("S&ettings"), this);
    settingsMenu->addAction(settingsConfigAction);
    this->menuBar()->addMenu(settingsMenu);

    this->menuBar()->addSeparator();


    // Help
    this->helpWebsiteAction = new QAction(QIcon::fromTheme("internet-web-browser"),
                                          tr("Visit Website"),
                                          this);
    connect(helpWebsiteAction, SIGNAL(triggered()),
            this, SLOT(visitWebsite()));

    this->helpBugTrackerAction = new QAction(QIcon::fromTheme("tools-report-bug"),
                                             tr("Report a Bug"),
                                             this);
    connect(helpBugTrackerAction, SIGNAL(triggered()),
            this, SLOT(visitBugTracker()));

    this->helpAboutXmppAction = new QAction(QIcon::fromTheme("internet-web-browser"),
                                            tr("About XMPP"),
                                            this);
    connect(helpAboutXmppAction, SIGNAL(triggered()),
            this, SLOT(visitXmppOrg()));

    this->helpAboutAction = new QAction(QIcon(":/icon/64x64/akarixb.png"),
                                        tr("About AkariXB"),
                                        this);
    connect(helpAboutAction, SIGNAL(triggered()),
            this, SLOT(aboutAkariXB()));

    this->helpMenu = new QMenu(tr("&Help"), this);
    helpMenu->addAction(helpWebsiteAction);
    helpMenu->addAction(helpBugTrackerAction);
    helpMenu->addAction(helpAboutXmppAction);
    helpMenu->addSeparator();
    helpMenu->addAction(helpAboutAction);
    this->menuBar()->addMenu(helpMenu);


    /// Context menu for the tray icon
    trayTitleSeparatorAction = new QAction("AkariXB", this);
    trayTitleSeparatorAction->setSeparator(true);

    trayShowWindowAction = new QAction(QIcon(":/icon/64x64/akarixb.png"),
                                       "*show-hide-window*",
                                       this);
    connect(trayShowWindowAction, &QAction::triggered,
            this, &MainWindow::toggleMainWindow);


    this->systrayMenu = new QMenu("*systray-menu*", this);
    systrayMenu->setSeparatorsCollapsible(false);
    systrayMenu->addAction(trayTitleSeparatorAction);
    systrayMenu->addAction(trayShowWindowAction);
    systrayMenu->addAction(sessionConnectAction);
    systrayMenu->addAction(sessionDisconnectAction);
    systrayMenu->addAction(helpAboutAction);
    systrayMenu->addSeparator();
    systrayMenu->addAction(sessionQuitAction);
}


void MainWindow::createTrayIcon()
{
    this->systrayIcon = new QSystemTrayIcon(QIcon(":/icon/64x64/akarixb.png"),
                                            this);

    if (systrayIcon->isSystemTrayAvailable())
    {
        //trayIconAvailable = true;

        // Catch clicks on icon
        connect(systrayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
                this, SLOT(trayControl(QSystemTrayIcon::ActivationReason)));

        // clicking in a popup notification (balloon-type) will show the window
        connect(systrayIcon, SIGNAL(messageClicked()),
                this, SLOT(show()));  // FIXME: this can mess up the first action in the context menu

        // Show notifications requested via GlobalObject
        connect(m_globalObject, &GlobalObject::showNotificationPopup,
                this, &MainWindow::showNotification);

        // Set contextual menu for the icon
        systrayIcon->setContextMenu(systrayMenu);
        systrayIcon->setToolTip("AkariXB");
        systrayIcon->show();

        qDebug() << "Tray icon created";
    }
    else
    {
        //trayIconAvailable = false;
        qDebug() << "System tray is NOT available";
    }
}



void MainWindow::createStatusBarWidgets()
{
    this->statusBarLabel = new QLabel(this);
    this->setStatusIconOnline(QXmppPresence::Online, false);
    connect(m_globalObject, &GlobalObject::statusChanged,
            this, &MainWindow::onStatusChanged);

    this->statusBar()->addPermanentWidget(statusBarLabel);
    this->statusBar()->setSizeGripEnabled(false);
}



void MainWindow::createDataDir()
{
    QString dataDirectory;
    dataDirectory = QStandardPaths::standardLocations(QStandardPaths::DataLocation)
                                   .first();

    m_globalObject->setDataDirectory(dataDirectory);
    qDebug() << "Data directory:" << dataDirectory;

    QDir dataDir;
    if (!dataDir.exists(dataDirectory))
    {
        qDebug() << "Creating data directory";
        if (dataDir.mkpath(dataDirectory))
        {
            m_globalObject->addToLog(tr("Data directory created: %1")
                                     .arg(dataDirectory));
            qDebug() << "Data directory created:" << dataDirectory;
        }
        else
        {
            m_globalObject->addToLog(tr("Error creating data directory: %1")
                                     .arg(dataDirectory));
            qDebug() << "Error creating data directory!";
        }
    }
    else
    {
        m_globalObject->addToLog(tr("Data directory in use: %1")
                                 .arg(dataDirectory));
    }

    if (!dataDir.exists(dataDirectory + "/logs"))
    {
        qDebug() << "Creating logs directory";
        if (dataDir.mkpath(dataDirectory + "/logs"))
        {
            qDebug() << "Logs directory created";
        }
        else
        {
            qDebug() << "Error creating logs directory!";
        }
    }
}



void MainWindow::setupXmppClient()
{
    m_xmppClient = new QXmppClient(this);
    m_xmppClient->versionManager().setClientName("AkariXB");
    m_xmppClient->versionManager().setClientOs("Akari-GNU-OS"); // tmp FIXME xD
    m_disconnectedManually = false;

    connect(m_xmppClient, &QXmppClient::connected,
            this, &MainWindow::onConnected);
    connect(m_xmppClient, &QXmppClient::disconnected,
            this, &MainWindow::onDisconnected);

    connect(m_xmppClient, &QXmppClient::messageReceived,
            this, &MainWindow::processMessage);
    connect(m_xmppClient, &QXmppClient::presenceReceived,
            this, &MainWindow::processPresence);
    connect(m_xmppClient, &QXmppClient::iqReceived,
            this, &MainWindow::processIq);

    connect(m_xmppClient, &QXmppClient::error,
            this, &MainWindow::onNetworkError);

    m_mucManager = new QXmppMucManager();
    m_xmppClient->addExtension(m_mucManager);

    m_receiptManager = new QXmppMessageReceiptManager();
    m_xmppClient->addExtension(m_receiptManager);


    m_globalObject->setXmppClient(m_xmppClient);
    m_globalObject->setMucManager(m_mucManager);
}


void MainWindow::enableXmppDebug()
{
    m_xmppClient->logger()->setLoggingType(QXmppLogger::StdoutLogging);
}




void MainWindow::loadSettings()
{
    QSettings settings;
    settings.beginGroup("Connection");
    m_userJid = settings.value("jid").toString();
    m_userPassword = QByteArray::fromBase64(settings.value("password")
                                                    .toByteArray());
    m_userAutoconnect = settings.value("autoconnect").toBool();
    m_userResource = settings.value("resource", QStringLiteral("AkariXB"))
                             .toString();
    m_userPriority = settings.value("priority", 10).toInt();
    settings.endGroup();


    settings.beginGroup("MainWindow");
    this->resize(settings.value("windowSize",
                                QSize(640, 500)).toSize());

    if (settings.contains("windowPosition"))
    {
        this->move(settings.value("windowPosition").toPoint());
    }
    settings.endGroup();
}


void MainWindow::saveSettings()
{
    QSettings settings;
    settings.beginGroup("Connection");
    settings.setValue("jid",         m_userJid);
    settings.setValue("password",    m_userPassword.toLocal8Bit().toBase64());
    settings.setValue("autoconnect", m_userAutoconnect);
    settings.setValue("resource",    m_userResource);
    settings.setValue("priority",    m_userPriority);
    settings.endGroup();


    settings.beginGroup("MainWindow");
    settings.setValue("windowSize",     this->size());
    settings.setValue("windowPosition", this->pos());
    settings.endGroup();
    settings.sync();
}


void MainWindow::setStatusIconOnline(QXmppPresence::AvailableStatusType statusType,
                                     bool isOnline)
{
    QString iconFile;

    if (isOnline)
    {
        switch (statusType)
        {
        case QXmppPresence::Online:
            iconFile = ":/images/button-online.png";
            break;

        case QXmppPresence::Away:
            iconFile = ":/images/button-away.png";
            break;

        case QXmppPresence::XA:
            iconFile = ":/images/button-xa.png";
            break;

        case QXmppPresence::DND:
            iconFile = ":/images/button-dnd.png";
            break;

        default:
            iconFile = ":/images/button-online.png";
            break;
        }
    }
    else
    {
        iconFile = ":/images/button-offline.png";
    }

    statusBarLabel->setPixmap(QPixmap(iconFile).scaled(16, 16));
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



void MainWindow::connectToServer(QString jid, QString password,
                                 bool autoconnect, QString resource,
                                 int priority)
{
    m_userJid = jid;
    m_userPassword = password;
    m_userAutoconnect = autoconnect;
    m_userResource = resource;
    m_userPriority = priority;

    const QString message = tr("Connecting...");
    this->generalModule->setInfoMessage(message);
    m_globalObject->addToLog(message);

    this->generalModule->setConnecting();
    this->sessionConnectAction->setDisabled(true);

    QXmppConfiguration clientConfig;
    clientConfig.setJid(m_userJid);
    clientConfig.setResource(m_userResource);
    clientConfig.setPassword(m_userPassword);
    clientConfig.setKeepAliveInterval(m_globalObject->getKeepAliveInterval());
    clientConfig.setKeepAliveTimeout(m_globalObject->getKeepAliveTimeout());
    clientConfig.setAutoReconnectionEnabled(false);

    m_disconnectedManually = false;
    m_initialConnectionTimer->start();

    m_xmppClient->connectToServer(clientConfig);
}


void MainWindow::autoConnectToServer()
{
    this->connectToServer(m_userJid,
                          m_userPassword,
                          m_userAutoconnect,
                          m_userResource,
                          m_userPriority);
}



void MainWindow::onConnected()
{
    const QString message = tr("%1 connected.").arg(m_userJid);
    this->generalModule->setInfoMessage(message);
    m_globalObject->addToLog(message);

    m_initialConnectionTimer->stop();

    QStringList statusMessages;    // TMP hardcoded 'default' messages
    statusMessages << "H{i|ii|iii}!"
                   << "I'm h{e|ee|eee}re{!|!!}"
                   << "%ownnick% is here{!|!!|!!!}"
                   << "I'm around...";

    QXmppPresence presence;
    presence.setPriority(m_userPriority);
    presence.setAvailableStatusType(QXmppPresence::Online);
    presence.setStatusText(this->variableParser
                               ->getParsed(Helpers::randomString(statusMessages)));
    m_xmppClient->setClientPresence(presence);

    // Need to disable this here again, for auto-reconnections
    this->sessionConnectAction->setDisabled(true);

    this->sessionDisconnectAction->setEnabled(true);

    this->roomModule->autojoinRooms();

    m_globalObject->setConnected(true);
    this->setTitleAndTrayInfo();
    this->setStatusIconOnline(QXmppPresence::Online, true);
}


void MainWindow::disconnectFromServer()
{
    m_initialConnectionTimer->stop();
    m_disconnectedManually = true;
    m_xmppClient->disconnectFromServer();
}


void MainWindow::onDisconnected()
{
    const QString message = tr("Disconnected.");
    this->generalModule->setInfoMessage(message);
    m_globalObject->addToLog(message);
    qDebug() << "\n"
                "\n=========================================================="
                "\n======================-DISCONNECTED-======================"
                "\n=========================================================="
                "\n";

    this->sessionDisconnectAction->setDisabled(true);
    this->sessionConnectAction->setEnabled(true);

    m_globalObject->setConnected(false);
    this->setTitleAndTrayInfo();
    this->setStatusIconOnline(QXmppPresence::Online, false);

    if (!m_disconnectedManually)
    {
        // We only notify the disconnection if it wasn't manual
        if (m_globalObject->getNotifyDisconnection())
        {
            m_globalObject->showNotification(tr("Disconnected"),
                                             tr("Connection to the server "
                                                "was lost."));
        }

        // TODO: optional email notification for this event

        this->autoConnectToServer(); // FIXME: autoreconnection should be optional
    }
}

/*
 * Workaround for https://github.com/qxmpp-project/qxmpp/issues/166
 *
 */
void MainWindow::onConnectionTimerAbort()
{
    m_globalObject->addToLog(tr("Connection seems stuck. Trying again..."));
    qDebug() << "==========================================================\n"
                "==============-ABORTING CONNECTION BY TIMER-==============\n"
                "==========================================================\n";
    m_initialConnectionTimer->stop();
    m_disconnectedManually = false;
    this->autoConnectToServer();
}


void MainWindow::onStatusChanged(QXmppPresence presence)
{
    this->setStatusIconOnline(presence.availableStatusType(),
                              m_globalObject->connectedToServer());
}


void MainWindow::processMessage(QXmppMessage message)
{
    // Send private messages to their tabs, first (order is important)
    if (message.type() == QXmppMessage::Chat
     || message.type() == QXmppMessage::Normal
     || message.type() == QXmppMessage::Headline) // TMP: This one could use its own place
    {
        // Ignore completely empty messages (from ChatState changes, etc)
        // FIXME: In the future, show these state changes somewhere else
        if (!message.body().isEmpty())
        {
            this->chatModule->addChatTab(message);
        }
    }


    // Afterwards, actually process the received message
    if (message.body().startsWith(m_globalObject->getCommandPrefix())
     && message.type() != QXmppMessage::Error)
    {
        this->commandHandler->processCommand(message);
    }
    else
    {
        this->messageHandler->processMessage(message);
    }

    //qDebug() << "processMessage() " << message.error().text()
    //             << message.error().code();
}


void MainWindow::processPresence(QXmppPresence presence)
{
    this->generalModule->addPresence(presence); // TMP...
}


void MainWindow::processIq(QXmppIq iq)
{
    qDebug() << QTime::currentTime().toString() << "IQ received:"
             << iq.from() << iq.to() << iq.type() << iq.id();

    foreach (QXmppElement iqExt, iq.extensions())
    {
        qDebug() << iqExt.tagName();
    }
}



void MainWindow::onNetworkError(QXmppClient::Error error)
{
    QString errorString;
    switch (error)
    {
    case QXmppClient::SocketError:
        errorString = tr("Socket Error")
                      + " (" + m_xmppClient->socketErrorString() + ")";
        break;

    case QXmppClient::KeepAliveError:
        errorString = tr("KeepAlive Error");
        break;

    case QXmppClient::XmppStreamError:
        errorString = tr("XMPP Stream Error");
        // If error is related to server shutdown, we might need to force reconnection
        qDebug() << "::: XMPP Stream Error:" << m_xmppClient->xmppStreamError();
        break;

    default:
        errorString = QStringLiteral("NoError");
    }


    const QString message = tr("Network error: %1").arg(errorString);
    m_globalObject->addToLog(message);

    //qDebug() << "Network error:"
    //         << "\nStanza:" << this->xmppClient->xmppStreamError()
    //         << "\nSocket:" << this->xmppClient->socketErrorString();
}


void MainWindow::setChatsTitle(int count)
{
    QString title = tr("Chats");
    if (count > 0)
    {
        // FIXME: this has some issues if the translated title is long
        title.prepend(QStringLiteral("* "));
        title.append(QString("\n--  %1  --").arg(count));
        // TODO: maybe hint with color, like:
        //// categoriesList->item(2)->setBackgroundColor(Qt::red);
        //// categoriesList->item(2)->setTextColor(Qt::white);
    }
    this->categoriesList->item(2)->setText(title);

    // FIXME: title bar needs updating when this changes!
}



void MainWindow::visitWebsite()
{
    qDebug() << "Opening website in browser";
    QDesktopServices::openUrl(QUrl("https://jancoding.wordpress.com/akarixb"));
}


void MainWindow::visitBugTracker()
{
    qDebug() << "Opening bugtracker in browser";
    QDesktopServices::openUrl(QUrl("https://gitlab.com/akarixb/akarixb-dev/issues"));
}


void MainWindow::visitXmppOrg()
{
    qDebug() << "Opening XMPP.org in browser";
    QDesktopServices::openUrl(QUrl("https://xmpp.org/about/technology-overview.html"));
}


/*
 * About... window
 *
 */
void MainWindow::aboutAkariXB()
{
    QMessageBox::about(this,
                       tr("About AkariXB"),
                       QString("<big><b>AkariXB v%1</b></big>")
                       .arg(qApp->applicationVersion())
                       + "<br />"
                         "Copyright 2015-2018  JanKusanagi<br />"
                         "<a href=\"https://jancoding.wordpress.com/akarixb\">"
                         "https://jancoding.wordpress.com/akarixb</a><br />"
                         "<br />"
                       + tr("AkariXB is a Jabber/XMPP bot.")
                       + "<br />"

                       + tr("You can talk to it, make it join chatrooms, etc.")

                       + "<hr>" // ---

                       + tr("English translation by JanKusanagi.",
                            "TRANSLATORS: Change this with your language "
                            "and name. If there was another translator before "
                            "you, add your name after theirs ;)")
                       + "<br>"
                         "<br>"

                       + tr("Thanks to all the testers, translators and "
                            "packagers, who help make AkariXB better!")

                       + "<hr>" // ---

                       + tr("AkariXB is Free Software, licensed under the "
                            "GNU GPL license, and uses some Oxygen icons "
                            "under LGPL license.")
                       + "<br><br>"

                         "<a href=\"http://www.gnu.org/licenses/gpl-2.0.html\">"
                         "GNU GPL v2</a> - "
                         "<a href=\"http://www.gnu.org/licenses/lgpl-2.1.html\">"
                         "GNU LGPL v2.1</a>"
                         "<br><br>"
                         "<a href=\"https://techbase.kde.org/Projects/Oxygen\">"
                         "techbase.kde.org/Projects/Oxygen</a>"
                         "<br>"
                         "<br>"

                       + QString("<b>Qt</b> v%1").arg(qVersion())
                       + QString(" &mdash; %1").arg(qApp->platformName())
                       + QString("<br>"
                                 "<b>QXMPP</b> v%1").arg(QXmppVersion()));
}


void MainWindow::setTitleAndTrayInfo()
{
    QString currentTabTitle = QStringLiteral("#");
    const QString offlineString = " (" + tr("Offline") + ")";

    if (this->categoriesList->currentRow() != -1)
    {
        currentTabTitle = this->categoriesList->currentItem()->text();
    }

    QString title = currentTabTitle;
    if (!m_userJid.isEmpty())
    {
        title.append(QString(" - %1/%2").arg(m_userJid, m_userResource));
    }

    if (m_globalObject->connectedToServer())
    {
        this->systrayIcon->setIcon(QIcon(":/icon/64x64/akarixb.png"));
    }
    else
    {
        this->systrayIcon->setIcon(QIcon(":/icon/64x64/akarixb-offline.png"));
        title.append(offlineString);
    }

    title.append(" - AkariXB");

    this->setWindowTitle(title);
    if (/*trayIconAvailable*/true) // FIXME
    {
        const QString idToShow = m_userJid.isEmpty()
                                 ? tr("Account is not configured")
                                 : m_userJid + "/" + m_userResource;
#ifdef Q_OS_UNIX
        title = "<b>AkariXB</b>"
                "<hr>"
                "<br>"
                + idToShow
                + (m_globalObject->connectedToServer() ?
                   QString() : offlineString);
#else
        // Some OSes don't render HTML in the tray tooltip
        title = "AkariXB: "
                "\n\n"
                + idToShow
                + (m_globalObject->connectedToServer() ?
                   QString() : offlineString);
#endif


        this->systrayIcon->setToolTip(title);

        this->trayTitleSeparatorAction->setText("AkariXB - " + m_userJid);
    }
}


void MainWindow::setStatusBarMessage(QString message)
{
    this->statusBar()->showMessage("[" + QTime::currentTime().toString() + "] "
                                   + message, 0);
}


void MainWindow::showNotification(QString title, QString message)
{
    if (/*trayIconAvailable*/true) // FIXME
    {
        this->systrayIcon->showMessage(title + " - AkariXB",
                                       message,
#if QT_VERSION < QT_VERSION_CHECK(5, 9, 0)
                                       QSystemTrayIcon::Information,
#else
                                       // This overload requires 5.9+
                                       QIcon::fromTheme("akarixb",
                                                        QIcon::fromTheme("dialog-information")),
#endif
                                       10000); // 10 secs
    }
}


/*
 * Control interaction with the system tray icon
 *
 */
void MainWindow::trayControl(QSystemTrayIcon::ActivationReason reason)
{
    qDebug() << "Tray icon activation reason:" << reason;

    if (reason != QSystemTrayIcon::Context) // Simple "main button" click in icon
    {
        // Hide or show the main window
        if (this->isMinimized())
        {
            // Hide and show, because raise() wouldn't work
            this->hide();
            this->showNormal();
            this->activateWindow();
            qDebug() << "RAISING from minimized state";
        }
        else
        {
            this->toggleMainWindow();
        }
    }
}


void MainWindow::toggleMainWindow(bool onStartup)
{
    bool shouldHide = false;
    if (onStartup)
    {
        shouldHide = m_globalObject->getHideInTray();
    }

    if (this->isHidden() && !shouldHide
        /*|| !this->trayIconAvailable*/)
    {
        this->trayShowWindowAction->setText(tr("&Hide Window"));
        this->show();
        this->activateWindow(); // Try to show it on top of others
        qDebug() << "SHOWING main window";
    }
    else
    {
        this->trayShowWindowAction->setText(tr("&Show Window"));
        this->hide();
        qDebug() << "HIDING main window";
    }
}



void MainWindow::quitProgram()
{
    if (this->isHidden())
    {
        this->toggleMainWindow();
    }

    // FIXME: make confirmation optional, with option of asking only if via tray
    int confirmation = QMessageBox::question(this,
                                             tr("Quit?") + " - AkariXB",
                                             tr("Are you sure?"),
                                             tr("Yes, quit AkariXB"), tr("No"),
                                             QString(), 1, 1);
    if (confirmation == 1)
    {
        return;
    }


    qApp->setQuitOnLastWindowClosed(true);

    std::cout << "\nShutting down AkariXB (" + m_userJid.toStdString()
                 + ")...\n";
    std::cout.flush();

    m_xmppClient->disconnectFromServer();
    qApp->closeAllWindows();

    std::cout << "All windows closed, bye! o/\n";
    std::cout.flush();

    qApp->quit();
}


//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


void MainWindow::resizeEvent(QResizeEvent *event)
{
    event->accept();
}


void MainWindow::closeEvent(QCloseEvent *event)
{
    qDebug() << "Trying to close window; hiding instead";
    this->toggleMainWindow(); // FIXME: only if tray icon was available
                              // Also, handle environment shutdown
    event->ignore();
}
