/*
 *   This file is part of AkariXB
 *   Copyright 2015-2018  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "globalobject.h"


GlobalObject::GlobalObject(QObject *parent) : QObject(parent)
{
    m_userConnected = false;
    m_connectionStartTime = QDateTime();

    // Load settings
    QSettings settings;
    settings.beginGroup("Configuration");

    this->syncGeneralSettings(settings.value("adminJids").toStringList(),
                              settings.value("commandPrefix",
                                             "!").toString(),
                              settings.value("keepAliveInterval",
                                             60).toInt(),
                              settings.value("keepAliveTimeout",
                                             50).toInt());


    this->syncNotificationSettings(settings.value("notifyNewChats",
                                                  false).toBool(),
                                   settings.value("notifyDisconnection",
                                                  false).toBool());


    this->syncSystraySettings(settings.value("hideInTray",
                                             false).toBool());



    qDebug() << "GlobalObject created";
}


GlobalObject::~GlobalObject()
{
    qDebug() << "GlobalObject destroyed";
}


void GlobalObject::syncGeneralSettings(QStringList adminJids,
                                       QString commandPrefix,
                                       int keepAliveInterval,
                                       int keepAliveTimeout)
{
    m_adminJids = adminJids;
    m_commandPrefix = commandPrefix;
    m_keepAliveInterval = keepAliveInterval;
    m_keepAliveTimeout = keepAliveTimeout;
}


QStringList GlobalObject::getAdminJids()
{
    return m_adminJids;
}


QString GlobalObject::getCommandPrefix()
{
    return m_commandPrefix;
}

int GlobalObject::getKeepAliveInterval()
{
    return m_keepAliveInterval;
}

int GlobalObject::getKeepAliveTimeout()
{
    return m_keepAliveTimeout;
}


void GlobalObject::syncNotificationSettings(bool forNewChats,
                                            bool forDisconnection)
{
    m_notifyNewChats = forNewChats;
    m_notifyDisconnection = forDisconnection;
}

bool GlobalObject::getNotifyNewChats()
{
    return m_notifyNewChats;
}

bool GlobalObject::getNotifyDisconnection()
{
    return m_notifyDisconnection;
}


void GlobalObject::syncSystraySettings(bool hideInTray)
{
    m_hideInSystemTray = hideInTray;
}

bool GlobalObject::getHideInTray()
{
    return m_hideInSystemTray;
}




void GlobalObject::setXmppClient(QXmppClient *client)
{
    m_xmppClient = client;
}

QXmppClient *GlobalObject::getXmppClient()
{
    return m_xmppClient;
}


void GlobalObject::setMucManager(QXmppMucManager *mucManager)
{
    m_xmppMucManager = mucManager;
}

QXmppMucManager *GlobalObject::getMucManager()
{
    return m_xmppMucManager;
}


void GlobalObject::setVariableParser(VariableParser *varParser)
{
    m_variableParser = varParser;
}

VariableParser *GlobalObject::getVariableParser()
{
    return m_variableParser;
}



void GlobalObject::setDataDirectory(QString path)
{
    m_dataDirectory = path;
}

QString GlobalObject::getDataDirectory()
{
    return m_dataDirectory;
}



void GlobalObject::addToLog(QString message, bool inStatusBar)
{
    emit logMessage(message);

    if (inStatusBar)
    {
        emit showInStatusBar(message);
    }
}


void GlobalObject::showNotification(QString title, QString message)
{
    emit showNotificationPopup(title, message);
}


void GlobalObject::setConnected(bool state)
{
    m_userConnected = state;
    if (m_userConnected)
    {
        m_connectionStartTime = QDateTime::currentDateTimeUtc();
    }
    else
    {
        m_connectionStartTime = QDateTime(); // Invalid time
    }
}


bool GlobalObject::connectedToServer()
{
    return m_userConnected;
}

QDateTime GlobalObject::getConnectionStartTime()
{
    return m_connectionStartTime;
}



void GlobalObject::addJoinedRoom(QString roomJid)
{
    if (!m_joinedRooms.contains(roomJid))
    {
        m_joinedRooms.append(roomJid);
    }
}

void GlobalObject::removeJoinedRoom(QString roomJid)
{
    m_joinedRooms.removeAll(roomJid);
}

QStringList GlobalObject::getJoinedRooms()
{
    return m_joinedRooms;
}


QXmppMucRoom *GlobalObject::roomFromJid(QString roomJid)
{
    foreach (QXmppMucRoom *room, m_xmppMucManager->rooms())
    {
        if (room->jid() == roomJid)
        {
            return room;
        }
    }

    return 0;
}


void GlobalObject::setOwnNickname(QString newNick)
{
    m_ownNickname = newNick;
    m_variableParser->setOwnNickname(newNick);
}


QString GlobalObject::getOwnNickname()
{
    return m_ownNickname;
}


void GlobalObject::notifyActivityChange(QString name, QString timeRange)
{
    QString text = name;
    if (!timeRange.isEmpty())
    {
        text.append(QString::fromUtf8("   \342\217\260 ") // Alarm clock symbol
                    + timeRange);
    }

    emit activityChanged(text);
}

void GlobalObject::notifyStatusChange(QXmppPresence presence)
{
    emit statusChanged(presence);
}



void GlobalObject::sendMessageToJid(QString jid, QString message)
{
    if (!jid.isEmpty() && !message.isEmpty())
    {
        if (this->getJoinedRooms().contains(jid))
        {
            QXmppMucRoom *room = this->roomFromJid(jid);
            if (room)
            {
                room->sendMessage(message);
            }
        }
        else
        {
            m_xmppClient->sendMessage(jid, message);
        }
    }
}



void GlobalObject::sendPrivateMessage(QString jid, QString message)
{
    m_xmppClient->sendMessage(jid, message);

    emit privateMessageSent(jid, message);
}

