﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGatewayRequest.h>
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace APIGateway {
namespace Model {

/**
 * <p>The GET request to get the usage data of a usage plan in a specified time
 * interval.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/GetUsageRequest">AWS
 * API Reference</a></p>
 */
class GetUsageRequest : public APIGatewayRequest {
 public:
  AWS_APIGATEWAY_API GetUsageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetUsage"; }

  AWS_APIGATEWAY_API Aws::String SerializePayload() const override;

  AWS_APIGATEWAY_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The Id of the usage plan associated with the usage data.</p>
   */
  inline const Aws::String& GetUsagePlanId() const { return m_usagePlanId; }
  inline bool UsagePlanIdHasBeenSet() const { return m_usagePlanIdHasBeenSet; }
  template <typename UsagePlanIdT = Aws::String>
  void SetUsagePlanId(UsagePlanIdT&& value) {
    m_usagePlanIdHasBeenSet = true;
    m_usagePlanId = std::forward<UsagePlanIdT>(value);
  }
  template <typename UsagePlanIdT = Aws::String>
  GetUsageRequest& WithUsagePlanId(UsagePlanIdT&& value) {
    SetUsagePlanId(std::forward<UsagePlanIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Id of the API key associated with the resultant usage data.</p>
   */
  inline const Aws::String& GetKeyId() const { return m_keyId; }
  inline bool KeyIdHasBeenSet() const { return m_keyIdHasBeenSet; }
  template <typename KeyIdT = Aws::String>
  void SetKeyId(KeyIdT&& value) {
    m_keyIdHasBeenSet = true;
    m_keyId = std::forward<KeyIdT>(value);
  }
  template <typename KeyIdT = Aws::String>
  GetUsageRequest& WithKeyId(KeyIdT&& value) {
    SetKeyId(std::forward<KeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The starting date (e.g., 2016-01-01) of the usage data.</p>
   */
  inline const Aws::String& GetStartDate() const { return m_startDate; }
  inline bool StartDateHasBeenSet() const { return m_startDateHasBeenSet; }
  template <typename StartDateT = Aws::String>
  void SetStartDate(StartDateT&& value) {
    m_startDateHasBeenSet = true;
    m_startDate = std::forward<StartDateT>(value);
  }
  template <typename StartDateT = Aws::String>
  GetUsageRequest& WithStartDate(StartDateT&& value) {
    SetStartDate(std::forward<StartDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ending date (e.g., 2016-12-31) of the usage data.</p>
   */
  inline const Aws::String& GetEndDate() const { return m_endDate; }
  inline bool EndDateHasBeenSet() const { return m_endDateHasBeenSet; }
  template <typename EndDateT = Aws::String>
  void SetEndDate(EndDateT&& value) {
    m_endDateHasBeenSet = true;
    m_endDate = std::forward<EndDateT>(value);
  }
  template <typename EndDateT = Aws::String>
  GetUsageRequest& WithEndDate(EndDateT&& value) {
    SetEndDate(std::forward<EndDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current pagination position in the paged result set.</p>
   */
  inline const Aws::String& GetPosition() const { return m_position; }
  inline bool PositionHasBeenSet() const { return m_positionHasBeenSet; }
  template <typename PositionT = Aws::String>
  void SetPosition(PositionT&& value) {
    m_positionHasBeenSet = true;
    m_position = std::forward<PositionT>(value);
  }
  template <typename PositionT = Aws::String>
  GetUsageRequest& WithPosition(PositionT&& value) {
    SetPosition(std::forward<PositionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of returned results per page. The default value is 25 and
   * the maximum value is 500.</p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline GetUsageRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_usagePlanId;

  Aws::String m_keyId;

  Aws::String m_startDate;

  Aws::String m_endDate;

  Aws::String m_position;

  int m_limit{0};
  bool m_usagePlanIdHasBeenSet = false;
  bool m_keyIdHasBeenSet = false;
  bool m_startDateHasBeenSet = false;
  bool m_endDateHasBeenSet = false;
  bool m_positionHasBeenSet = false;
  bool m_limitHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
