"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerNoteRoute = registerNoteRoute;
var _configSchema = require("@osd/config-schema");
var _notebooks = require("../../../common/constants/notebooks");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
var _saved_objects_notebooks_router = require("../../adaptors/notebooks/saved_objects_notebooks_router");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerNoteRoute(router) {
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const notebooksData = await opensearchNotebooksClient.find({
        type: _observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT,
        perPage: 1000
      });
      const fetchedNotebooks = (0, _saved_objects_notebooks_router.fetchNotebooks)(notebooksData.saved_objects);
      return response.ok({
        body: {
          data: fetchedNotebooks
        }
      });
    } catch (error) {
      console.log('Notebook:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    let notebooksData;
    try {
      const newNotebookObject = (0, _saved_objects_notebooks_router.createNotebook)(request.body);
      notebooksData = await opensearchNotebooksClient.create(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, newNotebookObject);
      return response.ok({
        body: `${notebooksData.id}`
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const notebookinfo = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.params.noteId);
      return response.ok({
        body: notebookinfo.attributes.savedNotebook
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/clone`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const getNotebook = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.body.noteId);
      const createCloneNotebook = (0, _saved_objects_notebooks_router.cloneNotebook)(getNotebook.attributes.savedNotebook, request.body.name);
      const createdNotebook = await opensearchNotebooksClient.create(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, createCloneNotebook);
      return response.ok({
        body: createdNotebook
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/rename`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const renamedNotebook = (0, _saved_objects_notebooks_router.renameNotebook)(request.body);
      const updatedNotebook = await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.body.noteId, renamedNotebook);
      return response.ok({
        body: updatedNotebook
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const deletedNotebooks = await opensearchNotebooksClient.delete(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.params.noteId);
      return response.ok({
        body: deletedNotebooks
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/addSampleNotebooks`,
    validate: {
      body: _configSchema.schema.object({
        visIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const sampleNotebooks = await (0, _saved_objects_notebooks_router.addSampleNotes)(opensearchNotebooksClient, request.body.visIds);
      return response.ok({
        body: sampleNotebooks
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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