/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { AuthType, debugLogger, OutputFormat } from '@google/gemini-cli-core';
import { USER_SETTINGS_PATH } from './config/settings.js';
import { validateAuthMethod } from './config/auth.js';
import {} from './config/settings.js';
import { handleError } from './utils/errors.js';
import { runExitCleanup } from './utils/cleanup.js';
function getAuthTypeFromEnv() {
    if (process.env['GOOGLE_GENAI_USE_GCA'] === 'true') {
        return AuthType.LOGIN_WITH_GOOGLE;
    }
    if (process.env['GOOGLE_GENAI_USE_VERTEXAI'] === 'true') {
        return AuthType.USE_VERTEX_AI;
    }
    if (process.env['GEMINI_API_KEY']) {
        return AuthType.USE_GEMINI;
    }
    return undefined;
}
export async function validateNonInteractiveAuth(configuredAuthType, useExternalAuth, nonInteractiveConfig, settings) {
    try {
        const effectiveAuthType = configuredAuthType || getAuthTypeFromEnv();
        const enforcedType = settings.merged.security?.auth?.enforcedType;
        if (enforcedType && effectiveAuthType !== enforcedType) {
            const message = effectiveAuthType
                ? `The enforced authentication type is '${enforcedType}', but the current type is '${effectiveAuthType}'. Please re-authenticate with the correct type.`
                : `The auth type '${enforcedType}' is enforced, but no authentication is configured.`;
            throw new Error(message);
        }
        if (!effectiveAuthType) {
            const message = `Please set an Auth method in your ${USER_SETTINGS_PATH} or specify one of the following environment variables before running: GEMINI_API_KEY, GOOGLE_GENAI_USE_VERTEXAI, GOOGLE_GENAI_USE_GCA`;
            throw new Error(message);
        }
        const authType = effectiveAuthType;
        if (!useExternalAuth) {
            const err = validateAuthMethod(String(authType));
            if (err != null) {
                throw new Error(err);
            }
        }
        await nonInteractiveConfig.refreshAuth(authType);
        return nonInteractiveConfig;
    }
    catch (error) {
        if (nonInteractiveConfig.getOutputFormat() === OutputFormat.JSON) {
            handleError(error instanceof Error ? error : new Error(String(error)), nonInteractiveConfig, 1);
        }
        else {
            debugLogger.error(error instanceof Error ? error.message : String(error));
            await runExitCleanup();
            process.exit(1);
        }
    }
}
//# sourceMappingURL=validateNonInterActiveAuth.js.map