/**************************************************************************
                          cmaproom.cpp
                      -------------------
    description          : The room map element
    begin                : Fri Oct 22 1999
    copyright            : (C) 1999 by Kmud Developer Team
    email                : kmud-devel@kmud.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmaproom.h"

void CMapRoom::paint(QPainter *p,CMapZone *currentZone)
{
	signed int y1,x1,x2,y2;
 	
	x1 = getLowX();
	y1 = getLowY();
	x2 = getHiX();
	y2 = getHiY();	

	// Draw the room
	p->setPen(white);
	p->drawLine(x1,y2,x1,y1);
	p->drawLine(x1,y1,x2-1,y1);
	p->setPen(black);
	p->drawLine(x1,y2,x2,y2);
	p->drawLine(x2,y2,x2,y1);
	if (login)
	{
		p->setBrush(loginCol);
		p->setPen(loginCol);
	}
	else
	{
		if (getUseDefaultCol())
		{
			p->setBrush(getDefaultColour());
			p->setPen(getDefaultColour());
	
			}
			else
			{
				p->setBrush(getColour());
				p->setPen(getColour());					
				
			}
	}			
	p->drawRect(x1+1,y1+1,ROOM_SIZE-3,ROOM_SIZE-3);

	// If this is the current room the user is in
	// then draw the symbol to show that.
	if (getCurrentRoom())
	{
		p->setPen( getCurrentColour() );
		p->setBrush( getCurrentColour() );
		p->drawEllipse(x1+4,y1+4,ROOM_SIZE - 9,ROOM_SIZE -9);
	}
	
	// If the room is selected then draw the selection indicator
	if (getSelected())
	{
		p->setPen(getSelectColour());
		p->setBrush(getSelectColour());
		p->drawRect(x2-3,y2-3,6,6);
	}

	for (CMapPath *path=pathList.first();path!=0;path = pathList.next())
	{
		if (path->getSrcDir() == UP)
		{
			p->setPen(black);
			p->setBrush(black);

			p->drawPoint(x1+4,y1+3);
			p->drawPoint(x1+3,y1+4);
			p->drawPoint(x1+4,y1+4);
			p->drawPoint(x1+5,y1+4);
		}

		if (path->getSrcDir() == DOWN)
		{
			p->setPen(black);
			p->setBrush(black);
		
			p->drawPoint(x1+4,y1+ROOM_SIZE-5);
			p->drawPoint(x1+3,y1+ROOM_SIZE-6);
			p->drawPoint(x1+4,y1+ROOM_SIZE-6);
			p->drawPoint(x1+5,y1+ROOM_SIZE-6);
		}
		
		if (path->getSrcDir() == SPECIAL)
		{
			p->setPen(specialCol);
			p->setBrush(specialCol);
			
			p->drawEllipse(x1+ROOM_SIZE-10,y1+5,5,ROOM_SIZE-10);			
		}
	}

}

void CMapRoom::lowerPaint(QPainter *p,CMapZone *currentZone)
{
	signed int y1,x1,x2,y2;

	
	x1 = getLowX()-5;
	y1 = getLowY()-5;
	x2 = getHiX()-5;
	y2 = getHiY()-5;	
	
	p->setPen(getLowerColour());
	QBrush brush(getLowerColour());
	brush.setStyle(Dense3Pattern);
	p->setBrush(brush);
	p->drawRect(x1+1,y1+1,ROOM_SIZE-2,ROOM_SIZE-2);
}

void CMapRoom::higherPaint(QPainter *p,CMapZone *currentZone)
{
	signed int y1,x1,x2,y2;	
	
	x1 = getLowX()+5;
	y1 = getLowY()+5;
	x2 = getHiX()+5;
	y2 = getHiY()+5;	
	
	p->setPen(getHigherColour());
	QBrush brush(getHigherColour());
	brush.setStyle(Dense7Pattern);
	p->setBrush(brush);
	p->drawRect(x1+1,y1+1,ROOM_SIZE-2,ROOM_SIZE-2);
}


void CMapRoom::dragPaint(QPainter *p,int offsetx,int offsety,CMapZone *currentZone)
{
	signed int y1,x1,x2,y2;
	
	x1 = getLowX() + offsetx;
	y1 = getLowY() + offsety;
	x2 = getHiX() + offsetx;
	y2 = getHiY() + offsety;
		
	p->setPen(black);
	p->setBrush(QColor(black));
	p->drawRect(x1+1,y1+1,ROOM_SIZE-2,ROOM_SIZE-2);	
	
	// If the room is selected then draw the selection indicator
	if (getSelected())
	{
		p->setPen(blue);
		p->setBrush(QColor(blue));
		p->drawRect(x2-3,y2-3,6,6);
	}

}

CMapRoom::CMapRoom(CMapZone *mapZone)
{	
	current = false;
	setSelected(false);

	login = false;
	zone = mapZone;

	xpos = 0;
	ypos = 0;
	
	label = "";
	description = "";
	useDefaultCol = true;
	
	beforeEnterCommand="";	
	afterEnterCommand="";
	beforeExitCommand="";
	afterExitCommand="";
	oldZone = NULL;
}

CMapRoom::~CMapRoom(void)
{
}

CMapElement *CMapRoom::copy(void)
{
	CMapRoom *room = new CMapRoom(getZone());
	
	room->setCurrentRoom(getCurrentRoom());
	room->setLevel(getLevel());
	room->setX(getX());
	room->setY(getY());
	room->setLowHiX(getLowX(),getHiX());
	room->setLowHiY(getLowY(),getHiY());
	room->setSelected(getSelected());
	room->setLowerColour(getLowerColour());
	room->setDefaultColour(getDefaultColour());
	room->setHigherColour(getHigherColour());
	room->setSelectColour(getSelectColour());
	room->setLoginColour(getLoginColour());
	room->setAfterEnterCommand(afterEnterCommand);
	room->setBeforeEnterCommand(beforeEnterCommand);	
	room->setBeforeExitCommand(beforeExitCommand);		
	room->setUseDefaultCol(useDefaultCol);
	room->setSpecialColour(specialCol);
	room->setColour(colour);
	room->setLabel(label);
	room->setCurrentColour(currentCol);
	room->setDescription(description);
	
	return room;
}

void CMapRoom::delPathReferences(void)
{
    connectingPaths.setAutoDelete(false);
    connectingPaths.clear();
    pathList.setAutoDelete(false);
    pathList.clear();
}

void CMapRoom::setX(int x)
{
	xpos = x;	
	
	int newLoX = (x * ROOM_SIZE) - ROOM_SIZE + 1;
	int newHiX = (x * ROOM_SIZE) - 1;

	setLowHiX(newLoX,newHiX);
}

void CMapRoom::setY(int y)
{
	ypos = y;
	
	int newLoY = (y * ROOM_SIZE) - ROOM_SIZE + 1;
	int newHiY = y * ROOM_SIZE-1;
	
	setLowHiY(newLoY,newHiY);
}

void CMapRoom::setEditMode(bool edit)
{
	editMode = false;
	if (edit)
		setSelected(false);
}

void CMapRoom::addPath (CMapPath *path)
{
	pathList.append(path);		
}

CMapPath *CMapRoom::getPathDirection (direction dir,QString specialCmd)
{
	CMapPath *path;
	if (dir!=SPECIAL)
	{
		for (path=pathList.first(); path!=0; path=pathList.next())
			if (path->getSrcDir()==dir)
				return path;
	}
	else
	{
		for (path=pathList.first(); path!=0; path=pathList.next())
			if (path->getSrcDir()==dir)
				if (path->getSpecialCmd()==specialCmd)
					return path;
	}


	return NULL;
}

void CMapRoom::setCurrentRoom(bool currentRoom)
{
   	current = currentRoom;
}

void CMapRoom::moveBy(int x, int y)	
{
	setX(getX()+(x / ROOM_SIZE));
	setY(getY()+(y / ROOM_SIZE));
}

void CMapRoom::read(KCMapFile* kcmfile)
{
  int x, y;
  useDefaultCol = kcmfile->readBoolEntry("use default color", true);
  setColour(kcmfile->readColorEntry("color"));
  x = kcmfile->readNumEntry("X", -1);
  y = kcmfile->readNumEntry("Y", -1);
  setX(x);
  setY(y);
  mapLevel = kcmfile->readNumEntry("level", 0);

  label = kcmfile->readEntry("label", "");
  description = kcmfile->readEntry("description", "");
  afterEnterCommand = kcmfile->readEntry("afterEnterCommand", "");
  beforeEnterCommand = kcmfile->readEntry("beforeEnterCommand", "");
  beforeExitCommand = kcmfile->readEntry("beforeExitCommand", "");
  afterExitCommand = kcmfile->readEntry("afterExitCommand", "");
}

void CMapRoom::write(KCMapFile* kcmfile)
{
  kcmfile->writeEntry("use default color", useDefaultCol);
  kcmfile->writeEntry("color", getColour());
  kcmfile->writeEntry("X", xpos);
  kcmfile->writeEntry("Y", ypos);
  kcmfile->writeEntry("level", mapLevel);

  kcmfile->writeEntry("label", label);
  kcmfile->writeEntry("description", description);
  kcmfile->writeEntry("afterEnterCommand", afterEnterCommand);
  kcmfile->writeEntry("beforeEnterCommand", beforeEnterCommand);
  kcmfile->writeEntry("beforeExitCommand", beforeExitCommand);
  kcmfile->writeEntry("afterExitCommand", afterExitCommand);
}
