#include "kpuissance4.h"
#include <kapp.h>
#include <kstdaccel.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <iostream.h>
#include <qmessagebox.h>
#include "resource.h"
#include "optionsdlg.h"

KPuissance4::KPuissance4(QWidget *parent, const char *name)
	: KTMainWindow(name), ServerProcess(0)
{
	readConfig(kapp->getConfig());
	initMenuBar();
	initToolBar();
	initStatusBar();
	initView();
	resize(400,500);
	MessageWindow->insertLine(i18n("Start server (if necessary), and connect to it"));
	setCaption("KPuissance4 Client");
}

KPuissance4::~KPuissance4()
{
	if (ServerProcess) delete ServerProcess;
}

void KPuissance4::readConfig(KConfig *conf)
{
	conf->setGroup("Configuration");
	BoardOptions.NumCols = conf->readNumEntry("NumCols",7);
	BoardOptions.NumRows = conf->readNumEntry("NumRows",6);
	BoardOptions.NumPuis = conf->readNumEntry("NumPuis",4);
	BoardOptions.UseBackgroundPixmap = conf->readBoolEntry("UseBackgroundPixmap",false);
	QString		defaultPixmap = (KApplication::kde_wallpaperdir() + "/circuit.jpg");
	BoardOptions.BackgroundPixmap = conf->readEntry("BackgroundPixmap",defaultPixmap.data());
	BoardOptions.BackgroundColor = conf->readColorEntry("BackgroundColor",&(colorGroup().base()));
	BoardOptions.Host = conf->readEntry("Host","localhost.localdomain");
	BoardOptions.Port = conf->readNumEntry("Port",2121);
	BoardOptions.ServerProgram = conf->readEntry("ServerProgram","server");
	BoardOptions.LocalMode = conf->readBoolEntry("LocalMode",false);
	BoardOptions.SocketName = conf->readEntry("SocketName","/tmp/kpuissance4");
	BoardOptions.Level = conf->readNumEntry("Level",1);
}

void KPuissance4::saveConfig(KConfig *conf)
{
	conf->setGroup("Configuration");
	conf->writeEntry("NumCols",BoardOptions.NumCols);
	conf->writeEntry("NumRows",BoardOptions.NumRows);
	conf->writeEntry("NumPuis",BoardOptions.NumPuis);
	conf->writeEntry("UseBackgroundPixmap",BoardOptions.UseBackgroundPixmap);
	conf->writeEntry("BackgroundPixmap",BoardOptions.BackgroundPixmap);
	conf->writeEntry("BackgroundColor",BoardOptions.BackgroundColor);
	conf->writeEntry("Host",BoardOptions.Host.data());
	conf->writeEntry("Port",BoardOptions.Port);
	conf->writeEntry("ServerProgram",BoardOptions.ServerProgram.data());
	conf->writeEntry("SocketName",BoardOptions.SocketName.data());
	conf->writeEntry("LocalMode",BoardOptions.LocalMode);
	conf->writeEntry("Level",BoardOptions.Level);
}

void KPuissance4::initMenuBar()
{
	KStdAccel	keys;

	QPopupMenu	*filePop = new QPopupMenu;
	filePop->insertItem(i18n("New game"),this,SLOT(newGame()),Key_N+CTRL,FILE_NEW_GAME_ID);
	filePop->insertItem(i18n("Open game"),this,SLOT(openGame()),keys.open(),FILE_OPEN_GAME_ID);
	filePop->insertItem(i18n("Save game"),this,SLOT(saveGame()),keys.save(),FILE_SAVE_GAME_ID);
	filePop->insertSeparator();
	filePop->insertItem(i18n("Quit"),this,SLOT(quitGame()),keys.quit(),FILE_EXIT_ID);
	menuBar()->insertItem(i18n("&File"),filePop);

	QPopupMenu	*actionPop = new QPopupMenu;
	actionPop->insertItem(i18n("Start server"),this,SLOT(startServer()),0,STARTSERVER_ID);
	actionPop->insertItem(i18n("Connect to server"),this,SLOT(connectToServer()),0,CONNECTTOSERVER_ID);
	actionPop->insertItem(i18n("Disconnect"),this,SLOT(disconnect()),0,DISCONNECT_ID);
	actionPop->insertItem(i18n("Configure..."),this,SLOT(configure()),Key_T+CTRL,CONFIGURE_ID);
	actionPop->insertSeparator();
	actionPop->insertItem(i18n("Create computer"),this,SLOT(createComputer()),0,COMPUTER_ID);
	actionPop->insertSeparator();
	actionPop->insertItem(i18n("Undo"),this,SLOT(undo()),Key_Z+CTRL,ACTION_UNDO_ID);
	actionPop->insertItem(i18n("Hint"),this,SLOT(hint()),0,ACTION_HINT_ID);
	menuBar()->insertItem(i18n("&Action"),actionPop);

	QPopupMenu	*optPop = new QPopupMenu;
	optPop->insertItem(i18n("Configure"),this,SLOT(configure()),Key_T+CTRL,CONFIGURE_ID);
	optPop->insertItem(i18n("Save options"),this,SLOT(saveOptions()),0,OPTIONS_SAVE_ID);
	menuBar()->insertItem(i18n("&Options"),optPop);

	QPopupMenu	*helpPop = kapp->getHelpMenu(true,"KPuissance4 v0.1\n\n"
							  "(C) 1999\n\n"
							  "Authors :\n"
							  "  AI  : Jean Bort (jbort@ufr.lirmm.fr)\n"
							  "  GUI : Michael Goffioul (goffioul@emic.ucl.ac.be)\n\n"
							  "Maintainer :\n"
							  "  Michael Goffioul");
	menuBar()->insertItem(i18n("&Help"),helpPop);
}

void KPuissance4::initToolBar()
{
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("exclamation.xpm"),STARTSERVER_ID,SIGNAL(clicked()),this,SLOT(startServer()),true,i18n("Start server"));
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("finish.xpm"),CONNECTTOSERVER_ID,SIGNAL(clicked()),this,SLOT(connectToServer()),true,i18n("Connect to server"));
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("start.xpm"),DISCONNECT_ID,SIGNAL(clicked()),this,SLOT(disconnect()),false,i18n("Disconnect"));
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("configure.xpm"),CONFIGURE_ID,SIGNAL(clicked()),this,SLOT(configure()),true,i18n("Options"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("reload.xpm"),FILE_NEW_GAME_ID,SIGNAL(clicked()),this,SLOT(newGame()),true,i18n("New game"));
//	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("fileopen.xpm"),FILE_OPEN_GAME_ID,SIGNAL(clicked()),this,SLOT(openGame()),true,i18n("Open game"));
//	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("filefloppy.xpm"),FILE_SAVE_GAME_ID,SIGNAL(clicked()),this,SLOT(saveGame()),true,i18n("Save game"));
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("computer.xpm"),COMPUTER_ID,SIGNAL(clicked()),this,SLOT(createComputer()),true,i18n("Create computer"));
//	toolBar()->insertSeparator();
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("undo.xpm"),ACTION_UNDO_ID,SIGNAL(clicked()),this,SLOT(undo()),true,i18n("Undo"));
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("idea.xpm"),ACTION_HINT_ID,SIGNAL(clicked()),this,SLOT(hint()),true,i18n("Hint"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("exit.xpm"),FILE_EXIT_ID,SIGNAL(clicked()),this,SLOT(quitGame()),true,i18n("Quit"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("help.xpm"),HELP_ID,SIGNAL(clicked()),this,SLOT(showHelp()),true,i18n("Help"));
}

void KPuissance4::initStatusBar()
{
	statusBar()->insertItem("",STATUSMSG_ID);
}

void KPuissance4::newGame()
{
	Board->restart();
}

void KPuissance4::saveGame()
{
	QMessageBox::critical(this,"Error","No effect for now.",QMessageBox::Ok | QMessageBox::Default,0);
}

void KPuissance4::openGame()
{
	QMessageBox::critical(this,"Error","No effect for now.",QMessageBox::Ok | QMessageBox::Default,0);
}

void KPuissance4::configure()
{
	OptionsDialog	*dlg = new OptionsDialog(this);
	dlg->setOptions(&BoardOptions,Board->connected() || ServerProcess != 0);
	if (dlg->exec()) {
		dlg->getOptions(&BoardOptions,Board->connected() || ServerProcess != 0);
		Board->updateBoard();
	}
	delete dlg;
}

void KPuissance4::undo()
{
	Board->undo();
}

void KPuissance4::hint()
{
	Board->hint();
}

void KPuissance4::quitGame()
{ close();}

void KPuissance4::showHelp()
{ kapp->invokeHTMLHelp("kpuissance4/index.html","");}

bool KPuissance4::queryClose()
{
	saveConfig(kapp->getConfig());
	if (Board->connected()) {
		QMessageBox::critical(this,i18n("Server error"),i18n("You are still connected, diconnected first"),QMessageBox::Ok | QMessageBox::Default,0);
		return false;
	}
	return true;
}

void KPuissance4::slotMessage(const char *text)
{
	statusBar()->changeItem(text,STATUSMSG_ID);
	QApplication::flushX();
}

void KPuissance4::initView()
{
	Splitter = new QSplitter(QSplitter::Vertical,this);
	Splitter->resize(400,500);
	Board = new PlayerBoard(&BoardOptions,Splitter);
	Board->resize(350,350);
	Splitter2 = new QSplitter(QSplitter::Vertical,Splitter);
	Splitter2->resize(350,150);
	MessageWindow = new QMultiLineEdit(Splitter2);
	MessageWindow->resize(350,125);
	Message = new QLineEdit(Splitter2);
	Message->resize(350,25);
	Message->setMinimumSize(20,25);
	Message->setMaximumSize(800,25);
	connect(Message,SIGNAL(returnPressed()),SLOT(messageSend()));
	setView(Splitter);
	Board->setMessageWindow(MessageWindow);
}

void KPuissance4::startServer()
{
	ServerProcess = new KProcess;
	connect(ServerProcess,SIGNAL(processExited(KProcess*)),SLOT(serverExited(KProcess*)));
	(*ServerProcess) << BoardOptions.ServerProgram.data();
	QString		str;
	str.setNum(BoardOptions.Port);
	if (BoardOptions.LocalMode) (*ServerProcess) << "-local" << BoardOptions.SocketName.data();
	else (*ServerProcess) << "-p" << str.data();
	str.setNum(BoardOptions.NumCols);
	(*ServerProcess) << "-nc" << str.data();
	str.setNum(BoardOptions.NumRows);
	(*ServerProcess) << "-nr" << str.data();
	str.setNum(BoardOptions.NumPuis);
	(*ServerProcess) << "-np" << str.data();
	ServerProcess->start(KProcess::NotifyOnExit,KProcess::NoCommunication);
	toolBar()->setItemEnabled(STARTSERVER_ID,false);
}

void KPuissance4::connectToServer()
{
	Board->connectToServer();
	if (Board->connected()) {
		toolBar()->setItemEnabled(STARTSERVER_ID,false);
		toolBar()->setItemEnabled(CONNECTTOSERVER_ID,false);
		toolBar()->setItemEnabled(DISCONNECT_ID,true);
	}
}

void KPuissance4::disconnect()
{
	Board->disconnect();
	toolBar()->setItemEnabled(STARTSERVER_ID,ServerProcess == 0);
	toolBar()->setItemEnabled(CONNECTTOSERVER_ID,true);
	toolBar()->setItemEnabled(DISCONNECT_ID,false);
}

void KPuissance4::serverExited(KProcess *proc)
{
	if (!proc->normalExit())
		QMessageBox::critical(this,i18n("Server error"),i18n("Server exited abnormally"),QMessageBox::Ok | QMessageBox::Default,0);
	delete ServerProcess;
	ServerProcess = 0;
	toolBar()->setItemEnabled(STARTSERVER_ID,true);
	toolBar()->setItemEnabled(CONNECTTOSERVER_ID,true);
}

void KPuissance4::messageSend()
{
	Board->messageToServer(Message->text());
	Message->clear();
}

void KPuissance4::saveOptions()
{
	saveConfig(kapp->getConfig());
	kapp->getConfig()->sync();
}

void KPuissance4::createComputer()
{
	Board->createComputerPlayer();
}