/***************************************************************************
 *   Copyright (C) 2008 by Jean-Baptiste Mardelle (jb@kdenlive.org)        *
 *   Copyright (C) 2011 by Marco Gittler (marco@gitma.de)                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          *
 ***************************************************************************/


#include "clipstabilize.h"
#include "effectstack/widgets/doubleparameterwidget.h"

#include <QDebug>
#include <mlt++/Mlt.h>
#include "kdenlivesettings.h"
#include <QFontDatabase>
#include <KMessageBox>
#include <KColorScheme>
#include <klocalizedstring.h>

ClipStabilize::ClipStabilize(const QStringList &urls, const QString &filterName,QWidget * parent) :
    QDialog(parent),
    m_filtername(filterName),
    m_urls(urls),
    vbox(NULL)
{
    setFont(QFontDatabase::systemFont(QFontDatabase::SmallestReadableFont));
    setupUi(this);
    setWindowTitle(i18n("Stabilize Clip"));
    auto_add->setText(i18np("Add clip to project", "Add clips to project", urls.count()));
    auto_add->setChecked(KdenliveSettings::add_new_clip());

    QPalette p = palette();
    KColorScheme scheme(p.currentColorGroup(), KColorScheme::View, KSharedConfig::openConfig(KdenliveSettings::colortheme()));
    QColor dark_bg = scheme.shade(KColorScheme::DarkShade);
    QColor selected_bg = scheme.decoration(KColorScheme::FocusColor).color();
    QColor hover_bg = scheme.decoration(KColorScheme::HoverColor).color();
    QColor light_bg = scheme.shade(KColorScheme::LightShade);

    QString stylesheet(QString("QProgressBar:horizontal {border: 1px solid %1;border-radius:0px;border-top-left-radius: 4px;border-bottom-left-radius: 4px;border-right: 0px;background:%4;padding: 0px;text-align:left center}\
                               QProgressBar:horizontal#dragOnly {background: %1} QProgressBar:horizontal:hover#dragOnly {background: %3} QProgressBar:horizontal:hover {border: 1px solid %3;border-right: 0px;}\
                               QProgressBar::chunk:horizontal {background: %1;} QProgressBar::chunk:horizontal:hover {background: %3;}\
                               QProgressBar:horizontal[inTimeline=\"true\"] { border: 1px solid %2;border-right: 0px;background: %4;padding: 0px;text-align:left center } QProgressBar::chunk:horizontal[inTimeline=\"true\"] {background: %2;}\
                               QAbstractSpinBox#dragBox {border: 1px solid %1;border-top-right-radius: 4px;border-bottom-right-radius: 4px;padding-right:0px;} QAbstractSpinBox::down-button#dragBox {width:0px;padding:0px;}\
                               QAbstractSpinBox::up-button#dragBox {width:0px;padding:0px;} QAbstractSpinBox[inTimeline=\"true\"]#dragBox { border: 1px solid %2;} QAbstractSpinBox:hover#dragBox {border: 1px solid %3;} ")
                               .arg(dark_bg.name()).arg(selected_bg.name()).arg(hover_bg.name()).arg(light_bg.name()));
                               setStyleSheet(stylesheet);

                               if (m_urls.count() == 1) {
                               QString newFile = m_urls.first();
                               newFile.append(".mlt");
                               QUrl dest(newFile);
                               dest_url->setMode(KFile::File);
                               dest_url->setUrl(QUrl(newFile));
} else {
                               label_dest->setText(i18n("Destination folder"));
                               dest_url->setMode(KFile::Directory | KFile::ExistingOnly);
                               dest_url->setUrl(QUrl(m_urls.first()).adjusted(QUrl::RemoveFilename));
}

                               if (m_filtername=="vidstab"){
                               // Some default params have to be set:
                               m_fixedParams.insert("algo", "1");
			       m_fixedParams.insert("relative", "1");
                               QStringList ls;
                               ls << "accuracy,type,int,value,8,min,1,max,10,tooltip,Accuracy of Shakiness detection";
                               ls << "shakiness,type,int,value,4,min,1,max,10,tooltip,How shaky is the Video";
                               ls << "stepsize,type,int,value,6,min,0,max,100,tooltip,Stepsize of Detection process minimum around";
                               //ls << "algo,type,bool,value,1,min,0,max,1,tooltip,0 = Bruteforce 1 = small measurement fields";
                               ls << "mincontrast,type,double,value,0.3,min,0,max,1,factor,1,decimals,2,tooltip,Below this Contrast Field is discarded";
                               //ls << "show,type,int,value,0,min,0,max,2,tooltip,0 = draw nothing. 1 or 2 show fields and transforms";
                               ls << "smoothing,type,int,value,10,min,0,max,100,tooltip,number of frames for lowpass filtering";
                               ls << "maxshift,type,int,value,-1,min,-1,max,1000,tooltip,max number of pixels to shift";
                               ls << "maxangle,type,double,value,-1,min,-1,max,3.14,decimals,2,tooltip,max angle to rotate (in rad)";
                               ls << "crop,type,bool,value,0,min,0,max,1,tooltip,0 = keep border  1 = black background";
                               //ls << "invert,type,bool,value,0,min,0,max,1,tooltip,invert transform";
                               //ls << "relative,type,bool,value,1,min,0,max,1,tooltip,0 = absolute transform  1= relative";
                               ls << "zoom,type,int,value,0,min,-500,max,500,tooltip,additional zoom during transform";
                               ls << "optzoom,type,bool,value,1,min,0,max,1,tooltip,use optimal zoom (calulated from transforms)";
                               ls << "sharpen,type,double,value,0.8,min,0,max,1,decimals,1,tooltip,sharpen transformed image";
                               fillParameters(ls);

}

                               connect(buttonBox->button(QDialogButtonBox::Ok), &QPushButton::clicked, this, &ClipStabilize::slotValidate);

                               vbox=new QVBoxLayout(optionsbox);
                               QHashIterator<QString,QHash<QString,QString> > hi(m_ui_params);
                               while(hi.hasNext()){
                               hi.next();
                               QHash<QString,QString> val=hi.value();
                               if (val["type"]=="int" || val["type"]=="double"){
        DoubleParameterWidget *dbl=new DoubleParameterWidget(hi.key(), val["value"].toDouble(),
                val["min"].toDouble(),val["max"].toDouble(),val["value"].toDouble(),
                "",0/*id*/,""/*suffix*/,val["decimals"]!=""?val["decimals"].toInt():0,this);
        dbl->setObjectName(hi.key());
        dbl->setToolTip(val["tooltip"]);
        connect(dbl, &DoubleParameterWidget::valueChanged, this, &ClipStabilize::slotUpdateParams);
        vbox->addWidget(dbl);
    }else if (val["type"]=="bool"){
                           QCheckBox *ch=new QCheckBox(hi.key(),this);
                           ch->setCheckState(val["value"] == "0" ? Qt::Unchecked : Qt::Checked);
                           ch->setObjectName(hi.key());
                           connect(ch, &QCheckBox::stateChanged, this, &ClipStabilize::slotUpdateParams);
                           ch->setToolTip(val["tooltip"]);
                           vbox->addWidget(ch);

                       }
}
adjustSize();
}

ClipStabilize::~ClipStabilize()
{
    /*if (m_stabilizeProcess.state() != QProcess::NotRunning) {
        m_stabilizeProcess.close();
    }*/
    KdenliveSettings::setAdd_new_clip(auto_add->isChecked());
}

QMap <QString, QString> ClipStabilize::producerParams()
{
    return QMap <QString, QString>();
}

QMap <QString, QString> ClipStabilize::filterParams()
{
    QMap <QString, QString> params;
    params.insert("filter", m_filtername);
    
    QMapIterator<QString, QString> i(m_fixedParams);
    while (i.hasNext()) {
	i.next();
	params.insert(i.key(), i.value());
    }

    QHashIterator <QString,QHash<QString,QString> > it(m_ui_params);
    while (it.hasNext()) {
        it.next();
	params.insert(it.key(), it.value().value("value"));
    }
    return params;
}

QMap <QString, QString> ClipStabilize::consumerParams()
{
    // consumer params
    QMap <QString, QString> params;
    params.insert("consumer", "xml");
    params.insert("all", "1");
    params.insert("title", i18n("Stabilised"));
    return params;
}

QString ClipStabilize::destination() const
{
    if (m_urls.count() == 1)
        return dest_url->url().path();
    else
        return dest_url->url().path() + QDir::separator();
}

QString ClipStabilize::desc() const
{
    return i18n("Stabilize clip");
}

void ClipStabilize::slotUpdateParams()
{
    for (int i=0;i<vbox->count();++i){
        QWidget* w=vbox->itemAt(i)->widget();
        QString name=w->objectName();
        if (!name.isEmpty() && m_ui_params.contains(name)){
            if (m_ui_params[name]["type"]=="int" || m_ui_params[name]["type"]=="double"){
                DoubleParameterWidget *dbl=static_cast<DoubleParameterWidget*>(w);
                m_ui_params[name]["value"]=QString::number((double)(dbl->getValue()));
            }else if (m_ui_params[name]["type"]=="bool"){
                QCheckBox *ch=(QCheckBox*)w;
                m_ui_params[name]["value"]= ch->checkState() == Qt::Checked ? "1" : "0" ;
            }
        }
    }
}

bool ClipStabilize::autoAddClip() const
{
    return auto_add->isChecked();
}

void ClipStabilize::fillParameters(QStringList lst)
{

    m_ui_params.clear();
    while (!lst.isEmpty()){
        QString vallist=lst.takeFirst();
        QStringList cont=vallist.split(',');
        QString name=cont.takeFirst();
        while (!cont.isEmpty()){
            QString valname=cont.takeFirst();
            QString val;
            if (!cont.isEmpty()){
                val=cont.takeFirst();
            }
            m_ui_params[name][valname]=val;
        }
    }

}

void ClipStabilize::slotValidate()
{
    if (m_urls.count() == 1) {
        if (QFile::exists(dest_url->url().path())) {
            if (KMessageBox::questionYesNo(this, i18n("File %1 already exists.\nDo you want to overwrite it?", dest_url->url().path() )) == KMessageBox::No) return;
        }
    }
    else {
        QDir folder(dest_url->url().toLocalFile());
        QStringList existingFiles;
        foreach(const QString &path, m_urls) {
            if (folder.exists(path + ".mlt")) existingFiles.append(folder.absoluteFilePath(path + ".mlt"));
        }
        if (!existingFiles.isEmpty()) {
            if (KMessageBox::warningContinueCancelList(this, i18n("The stabilize job will overwrite the following files:"), existingFiles) ==  KMessageBox::Cancel) return;
        }
    }
    accept();
}




