/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  DenomOperationMap,
  ExchangeFullDetails,
  ExchangeListItem,
  FeeDescriptionPair,
} from "@gnu-taler/taler-util";
import { ErrorAlertView } from "../../components/CurrentAlerts.js";
import { Loading } from "../../components/Loading.js";
import { ErrorAlert } from "../../context/alert.js";
import { HookError } from "../../hooks/useAsyncAsHook.js";
import { State as SelectExchangeState } from "../../hooks/useSelectedExchange.js";
import { ButtonHandler, SelectFieldHandler } from "../../mui/handlers.js";
import { compose, StateViewMap } from "../../utils/index.js";
import { useComponentState } from "./state.js";
import {
  ComparingView,
  NoExchangesView,
  PrivacyContentView,
  ReadyView,
  TosContentView,
} from "./views.js";

export interface Props {
  list: ExchangeListItem[];
  initialValue: string;
  onCancel: () => Promise<void>;
  onSelection: (exchange: string) => Promise<void>;
}

export type State =
  | State.Loading
  | State.LoadingUriError
  | State.Ready
  | State.Comparing
  | State.ShowingTos
  | State.ShowingPrivacy
  | SelectExchangeState.NoExchangeFound;

export namespace State {
  export interface Loading {
    status: "loading";
    error: undefined;
  }

  export interface LoadingUriError {
    status: "error";
    error: ErrorAlert;
  }

  export interface BaseInfo {
    exchanges: SelectFieldHandler;
    selected: ExchangeFullDetails;
    error: undefined;
    onShowTerms: ButtonHandler;
    onShowPrivacy: ButtonHandler;
  }

  export interface Ready extends BaseInfo {
    status: "ready";
    onClose: ButtonHandler;
  }

  export interface Comparing extends BaseInfo {
    status: "comparing";
    coinOperationTimeline: DenomOperationMap<FeeDescriptionPair[]>;
    wireFeeTimeline: Record<string, FeeDescriptionPair[]>;
    globalFeeTimeline: FeeDescriptionPair[];
    missingWireTYpe: string[];
    newWireType: string[];
    onReset: ButtonHandler;
    onSelect: ButtonHandler;
  }
  export interface ShowingTos {
    status: "showing-tos";
    exchangeUrl: string;
    onClose: ButtonHandler;
  }
  export interface ShowingPrivacy {
    status: "showing-privacy";
    exchangeUrl: string;
    onClose: ButtonHandler;
  }
}

const viewMapping: StateViewMap<State> = {
  loading: Loading,
  error: ErrorAlertView,
  comparing: ComparingView,
  "no-exchange-found": NoExchangesView,
  "showing-tos": TosContentView,
  "showing-privacy": PrivacyContentView,
  ready: ReadyView,
};

export const ExchangeSelectionPage = compose(
  "ExchangeSelectionPage",
  (p: Props) => useComponentState(p),
  viewMapping,
);
