// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "engine.h"
#include <cstdarg>

using namespace ost;
using namespace std;

BayonneSession **Bayonne::timeslots = NULL;
Bayonne::timeslot_t Bayonne::ts_count = 0;
Bayonne::timeslot_t Bayonne::ts_used = 0;
ScriptCommand *Bayonne::server = NULL;
const char *Bayonne::path_prompts;
const char *Bayonne::path_tmpfs;
const char *Bayonne::path_tmp;
std::ostream *Bayonne::logging = &cerr;
char *Bayonne::status = NULL;
timeout_t Bayonne::step_timer = 20;
timeout_t Bayonne::reset_timer = 60;
timeout_t Bayonne::exec_timer = 60000;
Mutex Bayonne::serialize;
char Bayonne::sla[32] = {0};
unsigned Bayonne::idle_count = 0;
unsigned Bayonne::idle_limit = 0;
bool Bayonne::shutdown_flag = false;
time_t Bayonne::start_time = 0;
AtomicCounter Bayonne::libexec_count = 0;
unsigned Bayonne::compile_count;
volatile bool Bayonne::image_loaded = false;
bool Bayonne::provision_ripple = true;
bool Bayonne::provision_daemon = false;
bool Bayonne::provision_system = false;
bool Bayonne::provision_check = false;
bool Bayonne::provision_user = false;
bool Bayonne::provision_test = false;
const char *Bayonne::provision_server = "bayonne";
const char *Bayonne::provision_libexec = "libexec";

Audio::Encoding Bayonne::peer_encoding = Audio::unknownEncoding;
Audio::timeout_t Bayonne::peer_framing = 0;
BayonneTranslator *Bayonne::init_translator = NULL;
const char *Bayonne::init_voicelib = "none/prompts";

Bayonne::statetab Bayonne::states[] = {
	{"initial", &BayonneSession::stateInitial, ' '},
	{"idle", &BayonneSession::stateIdle, '-'},
	{"reset", &BayonneSession::stateReset, '@'},
	{"release", &BayonneSession::stateRelease, '@'},
	{"busy", &BayonneSession::stateBusy, '^'},
	{"standby", &BayonneSession::stateStandby, '*'},
	{"ringing", &BayonneSession::stateRinging, '!'},
	{"pickup", &BayonneSession::statePickup, 'o'},
	{"seize", &BayonneSession::stateSeize, 'o'},
	{"run", &BayonneSession::stateRunning, 'x'},
	{"exec", &BayonneSession::stateLibexec, 'e'},
	{"thread", &BayonneSession::stateThreading, 't'},
	{"clear", &BayonneSession::stateClear, 'c'},
	{"inkey", &BayonneSession::stateInkey, 'c'},
	{"input", &BayonneSession::stateInput, 'c'},
	{"read", &BayonneSession::stateRead, 'c'},
	{"collect", &BayonneSession::stateCollect, 'c'},
	{"dial", &BayonneSession::stateDial, 'd'},
	{"xfer", &BayonneSession::stateXfer, 'd'},
	{"hold", &BayonneSession::stateHold, 'd'},
	{"recall", &BayonneSession::stateRecall, 'd'},
	{"tone", &BayonneSession::stateTone, 't'},
	{"dtmf", &BayonneSession::stateDTMF, 't'},
	{"play", &BayonneSession::statePlay, 'p'},
	{"record", &BayonneSession::stateRecord, 'r'},
	{"join", &BayonneSession::stateJoin, 'j'},
	{"accept", &BayonneSession::stateWait, 'w'},
	{"connect", &BayonneSession::stateConnect, 'd'},
	{"sleep", &BayonneSession::stateSleep, 's'},
	{"start", &BayonneSession::stateStart, 's'},
	{"hangup", &BayonneSession::stateHangup, 'h'},
	{"reset", &BayonneSession::stateLibreset, 'e'},
	{"keywait", &BayonneSession::stateWaitkey, 'c'}, 
	{"wait", &BayonneSession::stateLibwait, 's'},
	{"reset", &BayonneSession::stateIdleReset, '@'},
	{"final", &BayonneSession::stateFinal, ' '},
	{NULL, NULL, ' '}};

const char *Bayonne::getKeypath(const char *key)
{
	static char keypath[128];

        if(strnicmp(key + 1, "bayonne/", 8))
                return key;

        keypath[0] = *key;
        key += 9;
        snprintf(keypath + 1, sizeof(keypath) - 1, "%s/%s",
                provision_server, key);
        return keypath;
}    

void Bayonne::waitLoaded(void)
{
	while(!image_loaded)
	{
		Thread::sleep(100);
		Thread::yield();
	}
}

bool Bayonne::loadPlugin(const char *path)
{
        char pathbuf[256];
        const char *cp, *kv;
	const char *prefix = NULL;
        DSO *dso;

#ifdef	WIN32
#ifdef	HAVE_TESTING
        if(!prefix && provision_test)
#ifdef  _DEBUG
                prefix = "DEBUG";
#else
                prefix = "RELEASE";
#endif 
#endif
	if(!prefix)
		prefix = "C:\\Program Files\\Common Files\\GNU Telephony\\Bayonne Plugins";
#else
        char prefixbuf[256];

#ifdef	HAVE_TESTING
        if(!prefix && provision_test && !strchr(path, '/'))
        {
                snprintf(prefixbuf, sizeof(prefixbuf),
                        "%s/modules/%s/.libs", SOURCE_FILES, path);
                if(!isDir(prefixbuf))  
                        snprintf(prefixbuf, sizeof(prefixbuf),
                                "%s/modules/%s", SOURCE_FILES, path);
                prefix = prefixbuf;
        } 
#endif
	if(!prefix)
		prefix = LIBDIR_FILES;
#endif

#ifdef  WIN32
        snprintf(pathbuf, sizeof(pathbuf), "%s/%s." RLL_SUFFIX, prefix, path);
#else
        snprintf(pathbuf, sizeof(pathbuf), "%s/%s.dso", prefix, path);
#endif  
        cp = path;
        path = pathbuf;

        kv = server->getLast(path);
        if(kv)
        {
                if(!stricmp(kv, "loaded"))
                        return true;
		return false;
	}

        if(!canAccess(path))
        {
                errlog("access", "cannot load %s", path);
                return false;
        }

        dso = new DSO(path);
        if(!dso->isValid())
        {
                kv = dso->getError();
                server->setValue(path, kv);
                errlog("error", "cannot initialize %s", path);
                return false;
        }
        server->setValue(path, "loaded");
        return true;                
}

bool Bayonne::loadMonitor(const char *path)
{
        char pathbuf[256];
        const char *cp, *kv;
	const char *prefix = NULL;
        DSO *dso;

#ifdef	WIN32
#ifdef	HAVE_TESTING
        if(!prefix && provision_test)
#ifdef  _DEBUG
                prefix = "DEBUG";
#else
                prefix = "RELEASE";
#endif 
#endif
	if(!prefix)
		prefix = "C:\\Program Files\\Common Files\\GNU Telephony\\Bayonne Monitors";
#else
        char prefixbuf[256];

#ifdef	HAVE_TESTING
        if(!prefix && provision_test && !strchr(path, '/'))
        {
                snprintf(prefixbuf, sizeof(prefixbuf),
                        "%s/modules/%smon/.libs", SOURCE_FILES, path);
                if(!isDir(prefixbuf))  
                        snprintf(prefixbuf, sizeof(prefixbuf),
                                "%s/modules/%smon", SOURCE_FILES, path);
                prefix = prefixbuf;
        } 
#endif
	if(!prefix)
		prefix = LIBDIR_FILES;
#endif

#ifdef  WIN32
        snprintf(pathbuf, sizeof(pathbuf), "%s/%s." RLL_SUFFIX, prefix, path);
#else
        snprintf(pathbuf, sizeof(pathbuf), "%s/%s.mon", prefix, path);
#endif  
        cp = path;
        path = pathbuf;

        kv = server->getLast(path);
        if(kv)
        {
                if(!stricmp(kv, "loaded"))
                        return true;
		return false;
	}

        if(!canAccess(path))
        {
                errlog("access", "cannot load %s", path);
                return false;
        }

        dso = new DSO(path);
        if(!dso->isValid())
        {
                kv = dso->getError();
                server->setValue(path, kv);
                errlog("error", "cannot initialize %s", path);
                return false;
        }
        server->setValue(path, "loaded");
        return true;                
}

bool Bayonne::loadAudio(const char *cp)
{
	return false;
}

Bayonne::Handler Bayonne::getState(const char *id)
{
	unsigned pos = 0;
	while(states[pos].name)
	{
		if(!stricmp(states[pos].name, id))
			return states[pos].handler;
		++pos;
	}
	return (Handler)NULL;
}
	
bool Bayonne::matchDigits(const char *digits, const char *match, bool partial)
{
	unsigned len = strlen(match);
	unsigned dlen = 0;
	bool dflag = true, inc;
	const char *d = digits;
	char dbuf[32];

	if(*d == '+')
		++d;

	while(*d && dlen < sizeof(dbuf) - 1)
	{
		if(isdigit(*d) || *d == '*' || *d == '#')
		{
			dbuf[dlen++] = *(d++);
			continue;
		}

		if(*d == ' ' || *d == ',')
		{
			++d;
			continue;
		}

		if(*d == '!')
			break;

		if(!stricmp(digits, match))
			return true;

		return false;
	}	

	if(*d && *d != '!')
		return false;

	digits = dbuf;
	dbuf[dlen] = 0;	

	if(*match == '+')
	{
		++match;
		--len;
		if(dlen < len)
			return false;
		digits += (len - dlen);
	}

	while(*match)
	{
		inc = true;
		switch(*match)
		{
		case 'x':
		case 'X':
			if(!isdigit(*digits))
				return false;
			break;
		case 'N':
		case 'n':
			if(*digits < '2' || *digits > '9')
				return false;
			break;
		case 'O':
		case 'o':
			if(*digits && *digits != '1')
				inc = false;
			break;
		case 'Z':
		case 'z':
			if(*digits < '1' || *digits > '9')
				return false;
			break;
		case '?':
			if(!*digits)
				return false;
			break;
		default:
			if(*digits != *match)
				return false;
		}
		if(*digits && inc)
			++digits;
		++match;	
	}
	if(*digits)
		return partial;

	return true;
}

void Bayonne::allocate(timeslot_t max, ScriptCommand *cmd)
{
	if(timeslots)
		delete[] timeslots;

	if(status)
		delete[] status;

	if(cmd)
	{
		server = cmd;
		path_prompts = server->getLast("prompts");
		path_tmpfs = server->getLast("tmpfs");
		path_tmp = server->getLast("tmp");
	}

	status = new char[max + 1];
	timeslots = new BayonneSession*[max];
	memset(timeslots, 0, sizeof(BayonneSession*) * max);
	memset(status, 0x20, max);
	status[max] = 0;
	ts_count = max;
	ts_used = 0;
}

Bayonne::timeslot_t Bayonne::toTimeslot(const char *id)
{
	char buffer[16];
	char *cp;
	unsigned spid;
	timeslot_t ts;
	BayonneDriver *driver;
	BayonneSpan *span;
	BayonneSession *session;

	if(strchr(id, '-'))
	{
		ts = atoi(id);
		session = getSession(ts);
		if(!session)
			return NO_TIMESLOT;

		if(!stricmp(session->var_sid, id))
			return ts;

		return NO_TIMESLOT;
	}

	if(strchr(id, '+'))
	{
                ts = atoi(id);
                session = getSession(ts);
                if(!session)
                        return NO_TIMESLOT;

                if(!stricmp(session->var_tid, id))
                        return ts;

                return NO_TIMESLOT;
	}

	if(isdigit(*id))
	{
		ts = atoi(id);
		if(ts >= ts_used)
			return NO_TIMESLOT;
		return ts;
	}

	setString(buffer, sizeof(buffer), id);
	cp = strchr(buffer, '/');
	if(!cp)
		return NO_TIMESLOT;

	*(cp++) = 0;	
	driver = BayonneDriver::get(buffer);
	if(driver)
	{
		ts = atoi(cp);
		if(ts >= driver->getCount())
			return NO_TIMESLOT;

		return driver->getFirst() + ts;
	}
	spid = atoi(cp);
	cp = strchr(cp, ',');
	if(!cp)
		return NO_TIMESLOT;

	if(stricmp(buffer, "span"))
		return NO_TIMESLOT;

	ts = atoi(++cp);
	span = BayonneSpan::get(spid);
	if(!span)
		return NO_TIMESLOT;

	if(ts >= span->getCount())
		return NO_TIMESLOT;

	return span->getFirst() + ts;
}

BayonneSession *Bayonne::getSid(const char *id)
{
	timeslot_t ts = toTimeslot(id);
	return getSession(ts);
}

BayonneSession *Bayonne::getSession(timeslot_t timeslot)
{
	if(!timeslots)
		return NULL;

	if(timeslot >= ts_count)
		return NULL;

	return timeslots[timeslot];
}

bool Bayonne::service(const char *id)
{
	bool rtn = false;
	ScriptImage *img;
	Name *scr = NULL;

	if(!server)
		return false;

	server->enter();

	if(!stricmp(id, "up"))
	{
		sla[0] = 0;
		rtn = true;
		goto done;
	}

	img = server->getActive();
	if(!img)
		goto done;

	if(!strchr(id, ':'))
		scr = img->getScript(id);

	if(scr && scr->access != scrPUBLIC)
		scr = NULL;

	if(!scr)
	{
		slog.error("%s: unknown or invalid service level", id);
		goto done;
	}

	rtn = true;
	setString(sla, sizeof(sla), id);

done:
	server->leave();
	return rtn;
}

void Bayonne::down(void)
{
#ifndef	WIN32
	if(idle_count == idle_limit)
		raise(SIGTERM);
	else
		shutdown_flag = true;
#endif
}

ScriptCompiler *Bayonne::reload(void)
{
	static Mutex lock;
	const char *cp;
	char buffer[32];

	ScriptCompiler *img;

	if(!server)
		return NULL;

	compile_count = 0;

	lock.enter();
#ifdef	WIN32
	img = new ScriptCompiler(server, "/bayonne/config");
#else
	img = new ScriptCompiler(server, getKeypath("/bayonne/server/config"));
	if(provision_system)
		img->loadPrefix("config", "/bayonne/provision/config");
	else if(provision_user)
		img->loadPrefix("config", "~bayonne/config");
#endif

#ifdef  SCRIPT_DEFINE_TOKENS
        const char *const *list = server->getList("definitions");
        img->compileDefinitions("site.def");
        while(list && *list)
        {
                img->compileDefinitions(*list);
                ++list;
        }
	cp = server->getLast("location");
	if(cp)
	{
		snprintf(buffer, sizeof(buffer), "macros-%s.def", cp);
		img->compileDefinitions(buffer);
	}
	img->compileDefinitions("macros.def");
#endif

	ScriptBinder::rebuild(img);
	img->commit();
	lock.leave();
	if(compile_count)
		image_loaded = true;
	return img;
}

static char digit[16] = {
	'0', '1', '2', '3',
        '4', '5', '6', '7',
        '8', '9', '*', '#',
        'a', 'b', 'c', 'd'};

char Bayonne::getChar(int dig)
{
	if(dig < 0 || dig > 15)
		return 0;

	return digit[dig];
}

int Bayonne::getDigit(char dig)
{
        int i;

	static char digit[16] = {
        	'0', '1', '2', '3',
        	'4', '5', '6', '7',
        	'8', '9', '*', '#',
        	'a', 'b', 'c', 'd'};

        dig = tolower(dig);
        for(i = 0; i < 16; ++i)
        {
                if(digit[i] == dig)
                        return i;
        }
        return -1;
}

ScriptImage *Bayonne::useImage(void)
{
	ScriptImage *img;

	if(!server || !image_loaded)
		return NULL;

	server->enter();
	img = server->getActive();
	if(!img)
	{
		server->leave();
		return NULL;
	}
	img->incRef();
	server->leave();
	return img;
}

void Bayonne::endImage(ScriptImage *img)
{
	if(!img)
		return;

	server->enter();
	img->decRef();
	if(!img->isRef() && img != server->getActive())
		delete img;
        server->leave();
}

unsigned long Bayonne::uptime(void)
{
	time_t now;

	if(!start_time)
		return 0;

	time(&now);
	return now - start_time;
}

void Bayonne::errlog(const char *level, const char *fmt, ...)    
{
        char *m;
        char buffer[256];
        va_list args;

        va_start(args, fmt);
        vsnprintf(buffer, sizeof(buffer) - 1, fmt, args);

        m = strchr(buffer, '\n');
        if(m)
                *m = 0;
 
        if(!stricmp(level, "debug"))
        {
                slog.debug() << buffer << endl;
                return;
        } 
        else if(!stricmp(level, "missing"))
                slog.warn() << buffer << endl;
        else if(!stricmp(level, "access"))
                slog.warn() << buffer << endl;
        else if(!stricmp(level, "notice"))
                slog.notice() << buffer << endl;
        else if(!strnicmp(level, "warn", 4))
        {
                level = "warn";
                slog.warn() << buffer << endl;
        } 
        else if(!strnicmp(level, "crit", 4))
        {
                level = "fatal";
                slog.critical() << buffer << endl;
        }
        else
                slog.error() << buffer << endl;

	if(Bayonne::server) 	                                                 
		Bayonne::server->errlog(level, buffer);
}
