/*
** Datei: DVIIBM.C
** Autoren: Ingo Eichenseher, Gerhard Wilhelms, Hans-J. Tpfer
**
*/
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <conio.h>
#include <alloc.h>
#include <dos.h>
#include <bios.h>
#include <string.h>
#include "dvi.h"
#include "dvidvi.h"
#include "dviframe.h"
#include "dvihdcp.h"

#pragma warn -par

extern void ega_putline(long offset, int count, unsigned char huge *data);
extern void ega_normal(void);
extern void ncr_2plane( void far *temp, int count, void far *data );
extern void hgc_graphics( void );

typedef void (*vfuncptr_t)( long offset, int count, unsigned char huge *data );

static void init_screen();

static vfuncptr_t putline = ega_putline;

static int screen_lines = 350;
static int screen_bytes = 80;
static int new_mode;
static int old_mode;

static int pix_updown = 16;
static int pix_leftright = 1;
static int pix_pgupdown = 200;

static char save_screen[4000];

static unsigned char video_system = 0;
/*
 * System codes: 0  = none
 *               1  = MDA
 *               2  = CGA
 *               3  = EGA
 *               4  = MCGA
 *               5  = VGA
 *               6  = HGC (Hercules)
 *               7  = HGC+
 *               8  = Hercules InColor
 */
#define NONE    0
#define MDA     1
#define CGA     2
#define EGA     3
#define MCGA    4
#define VGA     5
#define HGC     6
#define HGCPLUS 7
#define INCOLOR 8

/* base color names */
#define BLACK       0
#define BLUE        1
#define GREEN       2
#define CYAN        3
#define RED     4
#define MAGENTA 5
#define BROWN       20
#define GRAY        7
#define YELLOW  62
#define WHITE       63
#define LIGHT       56
#define BRIGHT  64
#define DARK        -1

/* EGA color translation table  */
static const struct color_table{char *name; unsigned char nbr;}
    ega_colors[] =
    {{"BLACK", BLACK}, {"BLUE", BLUE}, {"GREEN", GREEN}, {"CYAN", CYAN},
    {"RED", RED}, {"MAGENTA", MAGENTA}, {"BROWN", BROWN},
    {"GRAY", GRAY}, {"YELLOW", YELLOW}, {"WHITE", WHITE}, {"LIGHT", LIGHT},
    {"BRIGHT", BRIGHT}, {"DARK", DARK}};

static char bgcolor[20] = "WHITE", fgcolor[20] = "BLACK";
static unsigned char fgc_nbr = BLACK, bgc_nbr = WHITE;

static char *sys[] =
{
    "none",
    "MDA",
    "CGA",
    "EGA",
    "MCGA",
    "VGA",
    "Hercules",
    "Hercules Plus",
    "Hercules InColor"
};

static unsigned char display_type = 0;
/*
 * Display types: 0  = none
 *                1  = MDA-compatible MONO
 *                2  = CGA-compatible COLOR
 *                3  = EGA-compatible COLOR
 *                4  = VGA-compatible MONO
 *                5  = VGA-compatible COLOR
 */
#define MONOLO   1
#define COLORLO  2
#define COLORHI  3
#define PS2MONO  4
#define PS2COLOR 5

static char *dis[] =
{
    "none",
    "MDA-compatible monochrome",
    "CGA-compatible color",
    "EGA-compatible color",
    "VGA-compatible monochrome",
    "VGA-compatible color"
};

static char *mod[] =
{
    "none",
    "640 x 200",
    "640 x 350",
    "640 x 480",
    "720 x 348"
};

typedef struct
{
    unsigned char columns;
    unsigned char mode;
    unsigned char page;
} video_status_t;

#define YSINC 0x57

static int checkcol(char *str1, char *str2)
/* to check if str1 ends with str2  *
 *  returns 1 if true, else 0           */
{
    int l1 = strlen(str1), l2 = strlen(str2);

    if ( (l1 >= l2) && !strcmpi(&str1[l1-l2], str2) )
    {
        str1[l1-l2] = '\0';
        return 1;
    }
    return 0;
}

static unsigned getcolnbr( const char *col_name)
{
    int i, primary_color = -1, secondary_color = 0;
    char name[20];
/*  return number directly, if given as such */
    if ( isdigit(*col_name) )
        return atoi(col_name);

/*  scan name from rear to get primary color name */
    strcpy(name, col_name);
    for ( i = 0; i < 10; i++ )
        if ( checkcol(name, ega_colors[i].name) )
        {
            primary_color = ega_colors[i].nbr;
            break;
        }
    switch ( primary_color )
    {
    case YELLOW:
        secondary_color = LIGHT;
        primary_color = 6;
        break;
    case BROWN:
        primary_color = RED;
        secondary_color = GREEN;
        break;
    case WHITE:
        secondary_color = (primary_color &= 7);
    }

/* scan rest of name from rear to get secondary color name */
    for ( i = 0; i < 13; i++ )
        if ( checkcol(name, ega_colors[i].name) )
            secondary_color = ega_colors[i].nbr;
    switch ( secondary_color )
    {
    case LIGHT:
        if ( primary_color != GRAY )
            return primary_color | LIGHT;
        else
            return GRAY;
    case DARK:
        return (primary_color<<3)&56;
    case BRIGHT:
        return ((primary_color<<3)&56) | primary_color;
    case WHITE:
        return primary_color | LIGHT;
    default:
        return ((secondary_color<<3)&56) | primary_color;
    }
}

static void setvideomode( int mode )
{
    if ( mode & 128 )
        hgc_graphics();
    else
    asm {
        mov ax,mode
        xor ah,ah
        int 10h
    }
    /* the following new for palette changing */
    if ( (mode & 127) >= 0x10 ) /* EGA or better */
    asm {
        mov ax,01000h;              /* set palette register 0 */
        mov bl,0
        mov bh,bgc_nbr
        int 10h
        mov ax,1000h;                   /* set palette register 15 */
        mov bl,0fh
        mov bh,fgc_nbr
        int 10h
    }
    /* set key for segment register access in TS4000 */
    if ( (mode & 127) >= 0x30 ) /* super VGA or better */
    {
    asm {
        mov dx,3bfh
        mov al,3
        out dx,al
        jmp aaa
    }
aaa:
    asm {
        mov dx,3d8h
        mov al,0a0h
        out dx,al
    }
    }
}

static int getvideomode( video_status_t *stat )
{
    asm {
        mov ah,0fh
        int 10h
        push    si
        les si,stat
        mov es:[si].columns,ah
        mov es:[si].mode,al
        mov es:[si].page,bh
        pop si
        xor ah,ah
    }
    return(_AX);
}

#define CRTADR    0xB800
#define ODDOFFSET 0x2000

void cga_putline( long offset, int count, unsigned char huge *data )
{
    int line = (int)(offset / screen_bytes);

    if ( line % 2 )
    {
        offset = ( offset - 80 ) / 2;
        memcpy( MK_FP( CRTADR, ODDOFFSET + offset ), (const void *)data,
                    count );
    }
    else
    {
        offset /= 2;
        memcpy( MK_FP( CRTADR, offset ), (const void *)data, count );
    }
}

#define CRT0 0xB800
#define CRT1 0xBC00
#define CRT2 0xA800
#define CRT3 0xAC00

void ncr_cga_putline( long offset, int count, unsigned char huge *data )
{
    unsigned char temp[160];
    int addr = (int)(( offset / 80 ) % 4);
    int offs = (int)(( offset / 320 ) * 160);

    ncr_2plane( (void far *)temp, count, (void far *)data );
    switch( addr )
    {
    case 0:
        memcpy( MK_FP( CRT0, offs ), (const void *)temp, 2 * count );
        break;
    case 1:
        memcpy( MK_FP( CRT1, offs ), (const void *)temp, 2 * count );
        break;
    case 2:
        memcpy( MK_FP( CRT2, offs ), (const void *)temp, 2 * count );
        break;
    case 3:
        memcpy( MK_FP( CRT3, offs ), (const void *)temp, 2 * count );
        break;
    }
}

#define HGCCRT 0xB000
#define HGCOFF1 0x2000
#define HGCOFF2 0x4000
#define HGCOFF3 0x6000

void hgc_putline(long offset, int count, unsigned char huge *data)
{
    int addr = (int)(( offset / 90 ) % 4);
    int offs = (int)(( offset / 360 ) * 90);

    switch( addr )
    {
    case 0:
        memcpy( MK_FP( HGCCRT, offs ), (const void *)data, count );
        break;
    case 1:
        memcpy( MK_FP( HGCCRT, HGCOFF1 + offs ), (const void *)data, count );
        break;
    case 2:
        memcpy( MK_FP( HGCCRT, HGCOFF2 + offs ), (const void *)data, count );
        break;
    case 3:
        memcpy( MK_FP( HGCCRT, HGCOFF3 + offs ), (const void *)data, count );
        break;
    }
}

#define CSCREEN 0xB800
#define CPG0 0x0000
#define CPG1 0x1000
#define CPG2 0x2000
#define CPG3 0x3000

#define MSCREEN 0xB000
#define MPG0 0x0000
#define MPG1 0x8000

void switch_graphic(int mode)
{
    static int cursor, screen_page;
    video_status_t vstate;

    if (mode)
    {
        old_mode = getvideomode( &vstate );
        screen_page = vstate.page;
        if ( ( old_mode & 7 ) == 7 )
            switch( screen_page )
            {
            case 0:
                memcpy( save_screen, MK_FP( MSCREEN, MPG0 ), 4000 );
                break;
            case 1:
                memcpy( save_screen, MK_FP( MSCREEN, MPG1 ), 4000 );
                break;
            }
        else
            switch( screen_page )
            {
            case 0:
                memcpy( save_screen, MK_FP( CSCREEN, CPG0 ), 4000 );
                break;
            case 1:
                memcpy( save_screen, MK_FP( CSCREEN, CPG1 ), 4000 );
                break;
            case 2:
                memcpy( save_screen, MK_FP( CSCREEN, CPG2 ), 4000 );
                break;
            case 3:
                memcpy( save_screen, MK_FP( CSCREEN, CPG3 ), 4000 );
                break;
            }
        asm {
            mov ah,03h      ; /* Cursorposition auslesen */
            mov bh,BYTE PTR screen_page
            int 10h
            mov cursor,dx
        }
        setvideomode( new_mode );
    }
    else
    {
        setvideomode( old_mode );
        if ( ( old_mode & 7 ) == 7 )
            switch( screen_page )
            {
            case 0:
                memcpy( MK_FP( MSCREEN, MPG0 ), save_screen, 4000 );
                break;
            case 1:
                memcpy( MK_FP( MSCREEN, MPG1 ), save_screen, 4000 );
                break;
            }
        else
            switch( screen_page )
            {
            case 0:
                memcpy( MK_FP( CSCREEN, CPG0 ), save_screen, 4000 );
                break;
            case 1:
                memcpy( MK_FP( CSCREEN, CPG1 ), save_screen, 4000 );
                break;
            case 2:
                memcpy( MK_FP( CSCREEN, CPG2 ), save_screen, 4000 );
                break;
            case 3:
                memcpy( MK_FP( CSCREEN, CPG3 ), save_screen, 4000 );
                break;
            }

        asm {
            mov ah,02               ; /* Cursorposition setzen */
            mov bh,BYTE PTR screen_page
            mov dx,cursor
            int 10h
        }
    }
}

int screen(void)
{
    int i, key, lastkey=0, bytes = screen_bytes;
    long o;
    unsigned long p;

    int screen_x, screen_y;

    if (bytes>frame_width) bytes = frame_width;
    screen_x = screen_y = 0;

    switch_graphic(1);
    while(lastkey == 0)
    {
        if (screen_x+screen_bytes>frame_width)
            screen_x = frame_width - screen_bytes;
        if (screen_y+screen_lines>frame_height)
            screen_y = frame_height - screen_lines;
        if (screen_x<0) screen_x=0;
        if (screen_y<0) screen_y=0;
        p = (long)screen_y*(long)frame_width + screen_x;
        for (i=0, o=0; i<screen_lines && i<frame_height;
                i++, o+=screen_bytes, p+=frame_width)
            putline(o,bytes,frame_ptr(p));

        key = bioskey(0);
        switch(key)
        {
        case 0x4800 : screen_y -= pix_updown; break;
        case 0x5000 : screen_y += pix_updown; break;
        case 0x4b00 : screen_x -= pix_leftright; break;
        case 0x4d00 : screen_x += pix_leftright; break;
        case 0x4900 : screen_y -= pix_pgupdown; break;
        case 0x5100 : screen_y += pix_pgupdown; break;
        case 0x4700 : screen_x = screen_y = 0; break;
        case 0x4f00 : screen_y = frame_height - screen_lines; break;
        case 0x7300 : screen_x = 0; break;
        case 0x7400 : screen_x = frame_width - screen_bytes; break;
        case 0x011b :
        case 0x316e :
        case 0x314e :
        case 0x1c0d : lastkey = key; break;
        case 0x1071 :
        case 0x1051 : lastkey = 0x011b; break;
        default : putch(7);
        }
    }

    switch_graphic(0);

    return lastkey==0x011b;
}

void prbyte(int c)
{
    biosprint(0,c,op.biosdev);
}

extern unsigned xms_init(void);
extern unsigned xms_alloc(int);
extern void xms_free(unsigned);
extern void xms_copy(unsigned long, unsigned, unsigned long, unsigned, unsigned long);

static int xms_handle = 0;
static int xms_blocks = 0;

void install(void)
{
    init_screen();
    xms_blocks = xms_init();
    if (xms_blocks>127)
    {
        xms_blocks -= 64;
        xms_handle = xms_alloc(xms_blocks);
    }
    printf("%dK XMS-Memory in use\n",xms_blocks,xms_handle);
}

void destall(void)
{
    if (xms_handle) xms_free(xms_handle);
}

void get_videosystem( void )
{
    int i;
    video_status_t vstate;
    unsigned char keep_byte;

/*
 * Identify VGA/MCGA/EGA
 */
    asm {
        mov ax,1a00h
        int 10h
    }
/*
 * True only for VGA/MCGA
 */
    asm {
        cmp al,1ah
        jne   check_EGA
        cmp bl,0ah
        jb      isVGA
        mov video_system,MCGA
        ret
    }
 isVGA:
    asm mov video_system,VGA;
    return;

/*
 * Identify EGA
 */
check_EGA:
    asm {
        mov ah,12h
        mov bl,10h
        int 10h
        cmp bl,10h
        je      check_CGA
        mov video_system,EGA
        cmp bh,0
        jne isPS2
        mov display_type,COLORHI
        ret
        }
isPS2:
        asm mov display_type,PS2MONO;
        return;

 /*
  * must be CGA, MDA or Hercules, identify monochrome system
  */
 check_CGA:
    getvideomode( &vstate );
    if ( vstate.mode != 7 )
    {
/*
 * Identify CGA by looking at CGA ports
 */
        (void)outp( 0x3d4, 0x0f );
        keep_byte = inp( 0x3d5 );
        (void)outp( 0x3d5, 0x63 );
        delay( 5 );
        if ( inp( 0x3d5 ) == 0x63 )
        {
            (void)outp( 0x3d5, keep_byte );
            video_system = CGA;
            display_type = COLORLO;
            return;
        }
        else
        {
            (void)outp( 0x3d5, keep_byte );
            display_type = COLORLO;
            return;
        }
    }

/*
 * Identify MDA/HGC
 */
    (void)outp( 0x3b4, 0xf );
    keep_byte = inp( 0x3b5 );
    (void)outp( 0x3b5, 0x63 );
    delay( 5 );
    if ( inp( 0x3b5 ) != 0x63 )
    {
        (void)outp( 0x3b5, keep_byte );
        display_type = MONOLO;
        return;
    }
    (void)outp( 0x3b5, keep_byte );

/*
 * Identify HGC
 */
    keep_byte = inp( 0x3ba );
    keep_byte &= 0x80;
    for ( i = 0; i < 1000; i++ )
    {
        if ( ( inp( 0x3ba ) & 0x80 ) != keep_byte )
        {
            video_system = HGC;
            break;
        }
    }

    if ( video_system != HGC )
    {
        video_system = MDA;
        display_type = MONOLO;
        return;
    }

/*
 * Identify HGC, HGC+, InColor
 */
    display_type = MONOLO;
    keep_byte = inp( 0x3ba );
    switch( keep_byte )
    {
    case 0x50:
        video_system = INCOLOR;
        display_type = PS2COLOR;
        break;
    case 0x10:
        video_system = HGCPLUS;
        break;
    }
}

void get_display( void )
{
/*
 * Get monitor type on EGA/VGA/MCGA
 */
    if ( video_system >= EGA && video_system <= VGA )
    {
        asm {
            mov ah,12h;
            mov bl,10h;
            int 10h;
        }

        if ( _BH == 1 )
            display_type = PS2MONO;
        else
            if ( ( _CL & 0x0f ) == 9 )
                display_type = COLORHI;
            else
                display_type = COLORLO;
    }
}

static void init_screen(void)
{
    unsigned char modus = 0;
/*
 * First look for Environment settings that will overrride internal
 * tests. Good method to destroy your computer if you do not know
 * what you are doing!
 *  After HIGH forground and background colors may be given in EGA
 * terminology. Default colors are WHITE on BLACK
 */
    char *userdef = getenv( "GRAFX" ), *cptr;

    if ( userdef != NULL )
    {
        cptr = strtok(userdef, " ,");
        if ( cptr == NULL )
            goto skip;
        new_mode = atoi( cptr );
        cptr = strtok( NULL, " ," );
        if ( cptr == NULL )
            goto skip;
        screen_lines = atoi( cptr );
        cptr = strtok( NULL, " ," );
        if ( cptr == NULL )
            goto skip;
        screen_bytes = atoi( cptr );
        cptr = strtok( NULL, " ," );
        if ( ! strcmp( cptr, "LOW" ) )
            putline = cga_putline;
        if ( ! strcmp( cptr, "MID" ) )
            putline = ncr_cga_putline;
        if ( ! strcmp( cptr, "MONO" ) )
        {
            new_mode |= 128;
            putline = hgc_putline;
        }
        if ( ! strcmp( cptr, "HIGH" ) ) /* new for colors */
        {
            putline = ega_putline;
            cptr = strtok( NULL, " ," );
            if ( cptr != NULL )
                strcpy( fgcolor, cptr );
            cptr = strtok( NULL, " ," );
            if ( cptr != NULL )
                strcpy( bgcolor, cptr );
            fgc_nbr = getcolnbr(fgcolor);
            bgc_nbr = getcolnbr(bgcolor);
            printf( "User-defined graphics mode %d/%d/%d/ running %s(%d) on %s(%d).\n",
                    new_mode, screen_lines, screen_bytes, fgcolor, fgc_nbr, bgcolor, bgc_nbr );
        }
        return;
    }

skip:

    get_videosystem();
    if ( ! display_type )
        get_display();
    if ( video_system == MDA )
    {
        putch( 7 );
        printf( "\nMDA system not able to display graphics. Exiting!\n" );
        putch( 7 );
        exit( 0 );
    }
    if ( video_system == HGC || video_system == HGCPLUS )
    {
        modus = 4;
        new_mode = 7 | 128;
        screen_lines = 348;
        screen_bytes = 90;
        putline = hgc_putline;
        goto report;
    }
    if ( video_system == CGA )
    {
        modus = 1;
        new_mode = 6;
        screen_lines = 200;
        putline = cga_putline;
        goto report;
    }
    modus = 2;
    screen_lines = 350;
    if ( display_type == PS2MONO )
    {
        new_mode = 0xf;
        goto report;
    }
    if ( video_system == EGA )
    {
        new_mode = 0x10;
        goto report;
    }
    modus = 3;
    new_mode = 0x12;
    screen_lines = 480;

report:

    printf( "Graphics system: %s, Display: %s, Video mode: %s\n",
                sys[video_system], dis[display_type], mod[modus] );
/*
 * CGA force for testing
 * new_mode = 6;
 * screen_lines = 200;
 * putline = cga_putline;
 */
}

int stop_key(void)
{
    return kbhit() && getch()==27;
}

void movetoe(byte huge *p, long address, long size)
{
    xms_copy(size,0,(unsigned long)p,xms_handle,address);
}

void movefrome(byte huge *p, long address, long size)
{
    xms_copy(size,xms_handle,address,0,(unsigned long)p);
}

long get_extended(void)
{
    if (xms_handle) return (long)xms_blocks*1024l;
    return 0l;
}
