/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.compaction.Verifier;
import org.apache.cassandra.db.lifecycle.SSTableSet;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.KeyIterator;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.sstable.format.SSTableWriter;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.concurrent.Refs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SSTableImporter {
    private static final Logger logger = LoggerFactory.getLogger(ColumnFamilyStore.class);
    private final ColumnFamilyStore cfs;

    public SSTableImporter(ColumnFamilyStore cfs) {
        this.cfs = cfs;
    }

    @VisibleForTesting
    synchronized List<String> importNewSSTables(Options options) {
        logger.info("Loading new SSTables for {}/{}: {}", new Object[]{this.cfs.keyspace.getName(), this.cfs.getTableName(), options});
        List<Pair<Directories.SSTableLister, String>> listers = this.getSSTableListers(options.srcPaths);
        HashSet<Descriptor> currentDescriptors = new HashSet<Descriptor>();
        for (SSTableReader sSTableReader : this.cfs.getSSTables(SSTableSet.CANONICAL)) {
            currentDescriptors.add(sSTableReader.descriptor);
        }
        ArrayList<String> failedDirectories = new ArrayList<String>();
        if (options.verifySSTables || options.verifyTokens) {
            block14: for (Pair<Directories.SSTableLister, String> pair : listers) {
                Directories.SSTableLister sSTableLister = (Directories.SSTableLister)pair.left;
                String dir = (String)pair.right;
                for (Map.Entry<Descriptor, Set<Component>> entry : sSTableLister.list(true).entrySet()) {
                    Descriptor descriptor = entry.getKey();
                    if (currentDescriptors.contains(entry.getKey())) continue;
                    try {
                        this.verifySSTableForImport(descriptor, entry.getValue(), options.verifyTokens, options.verifySSTables, options.extendedVerify);
                    }
                    catch (Throwable t) {
                        if (dir != null) {
                            logger.error("Failed verifying sstable {} in directory {}", new Object[]{descriptor, dir, t});
                            failedDirectories.add(dir);
                            continue block14;
                        }
                        logger.error("Failed verifying sstable {}", (Object)descriptor, (Object)t);
                        throw new RuntimeException("Failed verifying sstable " + descriptor, t);
                    }
                }
            }
        }
        HashSet<SSTableReader> hashSet = new HashSet<SSTableReader>();
        for (Pair<Directories.SSTableLister, String> pair : listers) {
            Directories.SSTableLister lister = (Directories.SSTableLister)pair.left;
            String dir = (String)pair.right;
            if (failedDirectories.contains(dir)) continue;
            HashSet<MovedSSTable> movedSSTables = new HashSet<MovedSSTable>();
            HashSet<SSTableReader> newSSTablesPerDirectory = new HashSet<SSTableReader>();
            for (Map.Entry<Descriptor, Set<Component>> entry : lister.list(true).entrySet()) {
                try {
                    Descriptor oldDescriptor = entry.getKey();
                    if (currentDescriptors.contains(oldDescriptor)) continue;
                    File targetDir = this.getTargetDirectory(dir, oldDescriptor, entry.getValue());
                    Descriptor newDescriptor = this.cfs.getUniqueDescriptorFor(entry.getKey(), targetDir);
                    this.maybeMutateMetadata(entry.getKey(), options);
                    movedSSTables.add(new MovedSSTable(newDescriptor, entry.getKey(), entry.getValue()));
                    SSTableReader sstable = SSTableReader.moveAndOpenSSTable(this.cfs, entry.getKey(), newDescriptor, entry.getValue(), options.copyData);
                    newSSTablesPerDirectory.add(sstable);
                }
                catch (Throwable t) {
                    newSSTablesPerDirectory.forEach(s -> s.selfRef().release());
                    if (dir != null) {
                        logger.error("Failed importing sstables in directory {}", (Object)dir, (Object)t);
                        failedDirectories.add(dir);
                        if (options.copyData) {
                            this.removeCopiedSSTables(movedSSTables);
                        } else {
                            this.moveSSTablesBack(movedSSTables);
                        }
                        movedSSTables.clear();
                        newSSTablesPerDirectory.clear();
                        break;
                    }
                    logger.error("Failed importing sstables from data directory - renamed sstables are: {}", movedSSTables);
                    throw new RuntimeException("Failed importing sstables", t);
                }
            }
            hashSet.addAll(newSSTablesPerDirectory);
        }
        if (hashSet.isEmpty()) {
            logger.info("No new SSTables were found for {}/{}", (Object)this.cfs.keyspace.getName(), (Object)this.cfs.getTableName());
            return failedDirectories;
        }
        logger.info("Loading new SSTables and building secondary indexes for {}/{}: {}", new Object[]{this.cfs.keyspace.getName(), this.cfs.getTableName(), hashSet});
        Throwable throwable = null;
        try (Refs refs = Refs.ref(hashSet);){
            this.cfs.getTracker().addSSTables(hashSet);
            for (SSTableReader reader : hashSet) {
                if (!options.invalidateCaches || !this.cfs.isRowCacheEnabled()) continue;
                this.invalidateCachesForSSTable(reader.descriptor);
            }
        }
        catch (Throwable throwable2) {
            Throwable throwable3 = throwable2;
            throw throwable2;
        }
        logger.info("Done loading load new SSTables for {}/{}", (Object)this.cfs.keyspace.getName(), (Object)this.cfs.getTableName());
        return failedDirectories;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private File getTargetDirectory(String srcPath, Descriptor descriptor, Set<Component> components) {
        if (srcPath == null) {
            return descriptor.directory;
        }
        File targetDirectory = null;
        SSTableReader sstable = null;
        try {
            sstable = SSTableReader.open(descriptor, components, this.cfs.metadata);
            targetDirectory = this.cfs.getDirectories().getLocationForDisk(this.cfs.diskBoundaryManager.getDiskBoundaries(this.cfs).getCorrectDiskForSSTable(sstable));
        }
        finally {
            if (sstable != null) {
                sstable.selfRef().release();
            }
        }
        return targetDirectory == null ? this.cfs.getDirectories().getWriteableLocationToLoadFile(new File(descriptor.baseFilename())) : targetDirectory;
    }

    private List<Pair<Directories.SSTableLister, String>> getSSTableListers(Set<String> srcPaths) {
        ArrayList<Pair<Directories.SSTableLister, String>> listers = new ArrayList<Pair<Directories.SSTableLister, String>>();
        if (!srcPaths.isEmpty()) {
            for (String path : srcPaths) {
                File dir = new File(path);
                if (!dir.exists()) {
                    throw new RuntimeException(String.format("Directory %s does not exist", path));
                }
                if (!Directories.verifyFullPermissions(dir, path)) {
                    throw new RuntimeException("Insufficient permissions on directory " + path);
                }
                listers.add(Pair.create(this.cfs.getDirectories().sstableLister(dir, Directories.OnTxnErr.IGNORE).skipTemporary(true), path));
            }
        } else {
            listers.add(Pair.create(this.cfs.getDirectories().sstableLister(Directories.OnTxnErr.IGNORE).skipTemporary(true), null));
        }
        return listers;
    }

    private void moveSSTablesBack(Set<MovedSSTable> movedSSTables) {
        for (MovedSSTable movedSSTable : movedSSTables) {
            if (!new File(movedSSTable.newDescriptor.filenameFor(Component.DATA)).exists()) continue;
            logger.debug("Moving sstable {} back to {}", (Object)movedSSTable.newDescriptor.filenameFor(Component.DATA), (Object)movedSSTable.oldDescriptor.filenameFor(Component.DATA));
            SSTableWriter.rename(movedSSTable.newDescriptor, movedSSTable.oldDescriptor, movedSSTable.components);
        }
    }

    private void removeCopiedSSTables(Set<MovedSSTable> movedSSTables) {
        logger.debug("Removing copied SSTables which were left in data directories after failed SSTable import.");
        for (MovedSSTable movedSSTable : movedSSTables) {
            if (!new File(movedSSTable.newDescriptor.filenameFor(Component.DATA)).exists()) continue;
            SSTableWriter.delete(movedSSTable.newDescriptor, movedSSTable.components);
        }
    }

    @VisibleForTesting
    void invalidateCachesForSSTable(Descriptor desc) {
        try (KeyIterator iter = new KeyIterator(desc, this.cfs.metadata());){
            while (iter.hasNext()) {
                DecoratedKey decoratedKey = (DecoratedKey)iter.next();
                this.cfs.invalidateCachedPartition(decoratedKey);
            }
        }
    }

    private void verifySSTableForImport(Descriptor descriptor, Set<Component> components, boolean verifyTokens, boolean verifySSTables, boolean extendedVerify) {
        SSTableReader reader = null;
        try {
            reader = SSTableReader.open(descriptor, components, this.cfs.metadata);
            Verifier.Options verifierOptions = Verifier.options().extendedVerification(extendedVerify).checkOwnsTokens(verifyTokens).quick(!verifySSTables).invokeDiskFailurePolicy(false).mutateRepairStatus(false).build();
            try (Verifier verifier = new Verifier(this.cfs, reader, false, verifierOptions);){
                verifier.verify();
            }
        }
        catch (Throwable t) {
            throw new RuntimeException("Can't import sstable " + descriptor, t);
        }
        finally {
            if (reader != null) {
                reader.selfRef().release();
            }
        }
    }

    private void maybeMutateMetadata(Descriptor descriptor, Options options) throws IOException {
        if (new File(descriptor.filenameFor(Component.STATS)).exists()) {
            if (options.resetLevel) {
                descriptor.getMetadataSerializer().mutateLevel(descriptor, 0);
            }
            if (options.clearRepaired) {
                descriptor.getMetadataSerializer().mutateRepairMetadata(descriptor, 0L, null, false);
            }
        }
    }

    public static class Options {
        private final Set<String> srcPaths;
        private final boolean resetLevel;
        private final boolean clearRepaired;
        private final boolean verifySSTables;
        private final boolean verifyTokens;
        private final boolean invalidateCaches;
        private final boolean extendedVerify;
        private final boolean copyData;

        public Options(Set<String> srcPaths, boolean resetLevel, boolean clearRepaired, boolean verifySSTables, boolean verifyTokens, boolean invalidateCaches, boolean extendedVerify, boolean copyData) {
            this.srcPaths = srcPaths;
            this.resetLevel = resetLevel;
            this.clearRepaired = clearRepaired;
            this.verifySSTables = verifySSTables;
            this.verifyTokens = verifyTokens;
            this.invalidateCaches = invalidateCaches;
            this.extendedVerify = extendedVerify;
            this.copyData = copyData;
        }

        public static Builder options(String srcDir) {
            return new Builder(Collections.singleton(srcDir));
        }

        public static Builder options(Set<String> srcDirs) {
            return new Builder(srcDirs);
        }

        public static Builder options() {
            return Options.options(Collections.emptySet());
        }

        public String toString() {
            return "Options{srcPaths='" + this.srcPaths + '\'' + ", resetLevel=" + this.resetLevel + ", clearRepaired=" + this.clearRepaired + ", verifySSTables=" + this.verifySSTables + ", verifyTokens=" + this.verifyTokens + ", invalidateCaches=" + this.invalidateCaches + ", extendedVerify=" + this.extendedVerify + ", copyData= " + this.copyData + '}';
        }

        static class Builder {
            private final Set<String> srcPaths;
            private boolean resetLevel = false;
            private boolean clearRepaired = false;
            private boolean verifySSTables = false;
            private boolean verifyTokens = false;
            private boolean invalidateCaches = false;
            private boolean extendedVerify = false;
            private boolean copyData = false;

            private Builder(Set<String> srcPath) {
                assert (srcPath != null);
                this.srcPaths = srcPath;
            }

            public Builder resetLevel(boolean value) {
                this.resetLevel = value;
                return this;
            }

            public Builder clearRepaired(boolean value) {
                this.clearRepaired = value;
                return this;
            }

            public Builder verifySSTables(boolean value) {
                this.verifySSTables = value;
                return this;
            }

            public Builder verifyTokens(boolean value) {
                this.verifyTokens = value;
                return this;
            }

            public Builder invalidateCaches(boolean value) {
                this.invalidateCaches = value;
                return this;
            }

            public Builder extendedVerify(boolean value) {
                this.extendedVerify = value;
                return this;
            }

            public Builder copyData(boolean value) {
                this.copyData = value;
                return this;
            }

            public Options build() {
                return new Options(this.srcPaths, this.resetLevel, this.clearRepaired, this.verifySSTables, this.verifyTokens, this.invalidateCaches, this.extendedVerify, this.copyData);
            }
        }
    }

    private static class MovedSSTable {
        private final Descriptor newDescriptor;
        private final Descriptor oldDescriptor;
        private final Set<Component> components;

        private MovedSSTable(Descriptor newDescriptor, Descriptor oldDescriptor, Set<Component> components) {
            this.newDescriptor = newDescriptor;
            this.oldDescriptor = oldDescriptor;
            this.components = components;
        }

        public String toString() {
            return String.format("%s moved to %s with components %s", this.oldDescriptor, this.newDescriptor, this.components);
        }
    }
}

